'use strict'

/*
|--------------------------------------------------------------------------
| Routes
|--------------------------------------------------------------------------
|
| Http routes are entry points to your web application. You can create
| routes for different URL's and bind Controller actions to them.
|
| A complete guide on routing is available here.
| http://adonisjs.com/docs/4.1/routing
|
*/

/** @type {typeof import('@adonisjs/framework/src/Route/Manager')} */
const Route = use('Route')

Route.on('/').render('welcome')

function methodNotAllowed({
  response
}) {
  response.status(405)
  response.json({
    status: 'error',
    message: 'Method Not Allowed',
  })
}

Route.group('Shared With Platforms - No Authentication', () => {
  // Create User
  Route.post('user', 'UserController.postUser')

  // Login
  Route.post('auth/login', 'AuthController.login')
  Route.any('auth/login', methodNotAllowed)

  // Recover Password
  Route.post('recover_password', 'RecoverPasswordController.step_1')
  Route.post('recover_password/code/validate', 'RecoverPasswordController.step_2')
  Route.post('recover_password/change', 'RecoverPasswordController.step_3')
  Route.any('recover_password', methodNotAllowed)

  // Upload
  Route.post('upload', 'UploadController.upload')
  Route.any('upload', methodNotAllowed)
  Route.post('destroy', 'UploadController.destroy')
  Route.any('destroy', methodNotAllowed)

  // Shared
  Route.get('zipe_code', 'SharedController.getZipCode')
  Route.any('zipe_code', methodNotAllowed)

  // Get latitude and longitude through address
  Route.get('address/lat_lng', 'SharedController.getLatLngThroughAddress')
  Route.any('address/lat_lng', methodNotAllowed)

  Route.post('send/sms', 'SharedController.sensdSms')
  Route.any('send/sms', methodNotAllowed)

  // Adjust later
  Route.get('schedule/driver-raffle', 'DriverRaffleTaskController.index')
  Route.get('schedule/status-invoice', 'IuguInvoiceStatusTaskController.index')
  Route.get('schedule/start', 'ScheduleController.start')

}).prefix('v2')

// APP
Route.group('APP - No Authentication', () => {
  // User
  Route.get('user/:id', 'RedoUserController.show')
  Route.put('user/avatar/:id', 'RedoUserController.updateAvatar')
  Route.put('user/driver/status/:id', 'RedoUserController.updateStatus')
  Route.get('user/driver/status/:id', 'RedoUserController.getDriverStatus')
  Route.put('user/driver/update/player/:userId', 'UserController.updateOneSignalPlayerId')
  Route.any('user', methodNotAllowed)

  // Orders
  Route.resource('orders', 'RedoOrderController').apiOnly()
  Route.get('order/items/:id', 'RedoOrderController.showItems')
  Route.post('orders/send/sms/:id', 'RedoOrderController.sendSms')
  Route.get('orders/pending/:id', 'RedoOrderController.getPendingOrder').middleware(['auth:sameDay'])
  Route.put('orders/finish/:id', 'RedoOrderController.finishOrder')
  Route.get('orders/driver/:idDriver', 'RedoOrderController.getOrdersDriver')
  Route.get('order/driver/:idDriver/view/:idOrder', 'RedoOrderController.getOrderView')
  Route.put('order/verification/code/collect_or_destination', 'OrderController.verificationCodeCollectAndDestination')
  
  Route.any('orders', methodNotAllowed)

  // Order Driver
  Route.get('driver-order/delivery-status', 'DriverOrderController.getDeliveryStatus')
  Route.get('driver-order/delivery-fail-status', 'DriverOrderController.getDeliveryFailStatus')
  Route.get('driver-order/:idOrder(\\d+)/status', 'DriverOrderController.getOrderDeliveryStatus')
  Route.get('driver-order/:idDriver(\\d+)/orders', 'DriverOrderController.getDriverOrders').middleware(['auth:sameDay'])
  Route.post('driver-order/delivery', 'DriverOrderController.postOrderDelivery')
  Route.put('/driver-order/:idOrder(\\d+)/confirm/:confirm(\\d+)', 'DriverOrderController.putConfirmDelivery').middleware(['auth:sameDay'])

}).prefix('app/v2')

// Portal
Route.group('Portal - With Authentication', () => {
  // User
  Route.get('user', 'UserController.getUser')
  Route.put('user', 'UserController.putUser')
  Route.get('user/all', 'UserController.getUserAll')
  Route.put('user/avatar', 'UserController.putProfilePicture')
  Route.post('user/driver/validate', 'UserController.postValidateDriver')
  Route.any('user', methodNotAllowed)

  // Dashboard
  Route.get('dashboard/data', 'DashboardController.getDashboardData')
  Route.get('dashboard/graphics', 'DashboardController.getDashboardGraphics')

  // Orders
  Route.get('order', 'OrderController.getOrder')
  Route.put('order', 'OrderController.putOrder')
  Route.post('order', 'OrderController.postOrder')
  Route.get('order/all', 'OrderController.getOrderAll')
  Route.get('order/view', 'OrderController.getOrderView')
  Route.get('order/historic', 'OrderController.getHistoric')
  Route.delete('order/:idOrder(\\d+)?', 'OrderController.deleteOrder')
  Route.get('order/value', 'OrderController.getOrderValue')
  Route.post('order/:idOrder(\\d+)?/invoice', 'OrderController.postOrderInvoice')
  Route.post('order/:idOrder(\\d+)?/refund', 'OrderController.refundInvoice')
  Route.any('get', methodNotAllowed)

  // Order Driver
  Route.get('driver-order/deliver-status', 'DriverOrderController.getDriverStatus')
  Route.get('driver-order/deliver-fail-status', 'DriverOrderController.getDriverFailStatus')
  Route.get('driver-order/:idDriver(\\d+)/orders', 'DriverOrderController.getDriverOrders')
  Route.put('/driver-order/:idOrder(\\d+)/confirm/:confirm(\\d+)', 'DriverOrderController.putConfirmDelivery')

  // Financial
  Route.get('financial/resume', 'FinancialController.getResumeReport')
  Route.get('financial/driver-ranking', 'FinancialController.getDriverRankReport')
  Route.get('financial/admin-invoice', 'FinancialController.getAdminInvoiceReport')

  Route.get('financial/driver-resume', 'FinancialController.getDriverResumeReport')
  Route.get('financial/driver-payment', 'FinancialController.getDriverPaymentReport')

  // UserBank
  Route.get('user-bank/list', 'UserBankController.getUserBankList')
  Route.put('user-bank/:id(\\d+)?', 'UserBankController.putUserBank')
  Route.post('user-bank', 'UserBankController.postUserBank')
  Route.delete('user-bank/:id(\\d+)?', 'UserBankController.deleteUserBank')


}).prefix('portal/v2').middleware(['auth:sameDay'])

module.exports = {
  Route
}
