#!/usr/bin/env python
"""
Script para testar as funcionalidades do painel administrativo Allure
"""
import os
import sys
import django

# Configurar Django antes de importar qualquer coisa
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')
django.setup()

from django.test import TestCase, Client
from django.contrib.auth.models import User
from django.urls import reverse

def test_admin_access():
    """Testa o acesso ao painel administrativo"""
    print("🔍 Testando acesso ao painel administrativo...")
    
    client = Client()
    
    # Testar acesso sem autenticação (deve redirecionar para login)
    response = client.get('/admin/')
    if response.status_code == 302:  # Redirecionamento para login
        print("✅ Redirecionamento para login funcionando")
    else:
        print("❌ Erro: Não redirecionou para login")
        return False
    
    return True

def test_admin_models():
    """Testa se os modelos estão registrados no admin"""
    print("\n🔍 Testando registro de modelos no admin...")
    
    from django.contrib import admin
    
    # Verificar se os modelos principais estão registrados
    models_to_check = [
        'Modelo', 'Banner', 'Categoria', 'FotoModelo', 'Servico',
        'TransacaoPagamento', 'BoostPagamento', 'PremiumPagamento'
    ]
    
    registered_models = []
    for model_name in models_to_check:
        try:
            # Tentar encontrar o modelo no admin
            model_admin = None
            for model, admin_class in admin.site._registry.items():
                if model.__name__ == model_name:
                    model_admin = admin_class
                    break
            
            if model_admin:
                registered_models.append(model_name)
                print(f"✅ {model_name} registrado no admin")
            else:
                print(f"❌ {model_name} não encontrado no admin")
        except Exception as e:
            print(f"❌ Erro ao verificar {model_name}: {e}")
    
    print(f"\n📊 Total de modelos registrados: {len(registered_models)}/{len(models_to_check)}")
    return len(registered_models) == len(models_to_check)

def test_modelo_actions():
    """Testa as ações personalizadas do ModeloAdmin"""
    print("\n🔍 Testando ações personalizadas do ModeloAdmin...")
    
    # Criar um superusuário para teste
    try:
        user = User.objects.create_superuser(
            username='admin_test',
            email='admin@test.com',
            password='test123'
        )
        print("✅ Superusuário de teste criado")
    except:
        user = User.objects.get(username='admin_test')
        print("✅ Superusuário de teste já existe")
    
    # Verificar se as ações estão definidas
    from models.admin import ModeloAdmin
    expected_actions = [
        'aprovar_modelos', 'rejeitar_modelos', 'suspender_modelos',
        'ativar_premium', 'desativar_premium', 'ativar_boost', 'desativar_boost'
    ]
    
    available_actions = ModeloAdmin.actions
    missing_actions = [action for action in expected_actions if action not in available_actions]
    
    if missing_actions:
        print(f"❌ Ações faltando: {missing_actions}")
        return False
    else:
        print("✅ Todas as ações personalizadas estão implementadas")
        return True

def test_banner_actions():
    """Testa as ações personalizadas do BannerAdmin"""
    print("\n🔍 Testando ações personalizadas do BannerAdmin...")
    
    from models.admin import BannerAdmin
    expected_actions = ['ativar_banners', 'desativar_banners']
    
    available_actions = BannerAdmin.actions
    missing_actions = [action for action in expected_actions if action not in available_actions]
    
    if missing_actions:
        print(f"❌ Ações faltando: {missing_actions}")
        return False
    else:
        print("✅ Todas as ações de banner estão implementadas")
        return True

def test_admin_dashboard():
    """Testa o dashboard personalizado"""
    print("\n🔍 Testando dashboard personalizado...")
    
    # Verificar se o template existe
    template_path = 'templates/admin/index.html'
    if os.path.exists(template_path):
        print("✅ Template do dashboard existe")
    else:
        print("❌ Template do dashboard não encontrado")
        return False
    
    # Verificar se o AdminSite personalizado está configurado
    try:
        from allure_project.admin_site import admin_site
        print("✅ AdminSite personalizado configurado")
    except ImportError as e:
        print(f"❌ Erro ao importar AdminSite personalizado: {e}")
        return False
    
    return True

def test_admin_styling():
    """Testa se o CSS personalizado está funcionando"""
    print("\n🔍 Testando estilização personalizada...")
    
    # Verificar se o template base personalizado existe
    template_path = 'templates/admin/base_site.html'
    if os.path.exists(template_path):
        print("✅ Template base personalizado existe")
        
        # Verificar se contém estilos personalizados
        with open(template_path, 'r') as f:
            content = f.read()
            if '--primary-color' in content and 'dashboard-stats' in content:
                print("✅ Estilos personalizados encontrados")
                return True
            else:
                print("❌ Estilos personalizados não encontrados")
                return False
    else:
        print("❌ Template base personalizado não encontrado")
        return False

def main():
    """Função principal de teste"""
    print("🎭 TESTE DO PAINEL ADMINISTRATIVO ALLURE")
    print("=" * 50)
    
    # Executar testes
    tests = [
        ("Acesso ao Admin", test_admin_access),
        ("Registro de Modelos", test_admin_models),
        ("Ações do ModeloAdmin", test_modelo_actions),
        ("Ações do BannerAdmin", test_banner_actions),
        ("Dashboard Personalizado", test_admin_dashboard),
        ("Estilização Personalizada", test_admin_styling),
    ]
    
    results = []
    for test_name, test_func in tests:
        try:
            result = test_func()
            results.append((test_name, result))
        except Exception as e:
            print(f"❌ Erro no teste {test_name}: {e}")
            results.append((test_name, False))
    
    # Relatório final
    print("\n" + "=" * 50)
    print("📊 RELATÓRIO FINAL")
    print("=" * 50)
    
    passed = sum(1 for _, result in results if result)
    total = len(results)
    
    for test_name, result in results:
        status = "✅ PASSOU" if result else "❌ FALHOU"
        print(f"{test_name}: {status}")
    
    print(f"\n🎯 Resultado: {passed}/{total} testes passaram")
    
    if passed == total:
        print("🎉 Todos os testes passaram! Painel administrativo funcionando corretamente.")
    else:
        print("⚠️ Alguns testes falharam. Verifique as implementações.")
    
    return passed == total

if __name__ == "__main__":
    main() 