"""
Views para o sistema de busca avançada
"""

from django.shortcuts import render
from django.http import JsonResponse
from django.core.paginator import Paginator
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods
from django.contrib.auth.decorators import login_required
from django.utils.decorators import method_decorator
from django.views import View
from django.db.models import Q, Count
import json

from .forms import AdvancedSearchForm, QuickSearchForm, SearchSuggestionForm
from .engine import AdvancedSearchEngine, SearchSuggestionEngine, SearchAnalytics


def advanced_search(request):
    """
    View principal de busca avançada
    """
    # Inicializar formulário
    form = AdvancedSearchForm(request.GET or None)
    
    # Inicializar motor de busca
    search_engine = AdvancedSearchEngine()
    
    # Dados para contexto
    context = {
        'form': form,
        'page_obj': None,
        'search_stats': None,
        'filters_applied': {},
        'total_results': 0,
    }
    
    # Se o formulário for válido, executar busca
    if form.is_valid():
        # Obter filtros do formulário
        search_filters = form.get_search_filters()
        
        # Executar busca
        queryset, search_stats = search_engine.search(search_filters)
        
        # Aplicar paginação
        paginator = Paginator(queryset, search_filters.por_pagina)
        page_number = request.GET.get('page', 1)
        page_obj = paginator.get_page(page_number)
        
        # Atualizar contexto
        context.update({
            'page_obj': page_obj,
            'search_stats': search_stats,
            'filters_applied': search_stats.get('filters_applied', {}),
            'total_results': search_stats.get('total_results', 0),
        })
    
    # Adicionar dados para filtros
    context.update({
        'estados_disponiveis': _get_estados_disponiveis(),
        'cidades_populares': _get_cidades_populares(),
        'categorias_disponiveis': _get_categorias_disponiveis(),
    })
    
    return render(request, 'search/advanced_search.html', context)


def quick_search(request):
    """
    View de busca rápida
    """
    form = QuickSearchForm(request.GET or None)
    
    context = {
        'form': form,
        'page_obj': None,
        'total_results': 0,
    }
    
    if form.is_valid():
        # Converter para busca avançada
        search_filters = _convert_quick_to_advanced(form.cleaned_data)
        
        # Executar busca
        search_engine = AdvancedSearchEngine()
        queryset, search_stats = search_engine.search(search_filters)
        
        # Paginação
        paginator = Paginator(queryset, 20)
        page_number = request.GET.get('page', 1)
        page_obj = paginator.get_page(page_number)
        
        context.update({
            'page_obj': page_obj,
            'total_results': search_stats.get('total_results', 0),
        })
    
    return render(request, 'search/quick_search.html', context)


@csrf_exempt
@require_http_methods(["POST"])
def search_suggestions(request):
    """
    View AJAX para sugestões de busca
    """
    try:
        data = json.loads(request.body)
        term = data.get('term', '').strip()
        
        if len(term) < 2:
            return JsonResponse({'suggestions': []})
        
        # Obter sugestões
        suggestion_engine = SearchSuggestionEngine()
        suggestions = suggestion_engine.get_suggestions(term, limit=8)
        
        return JsonResponse({
            'suggestions': suggestions,
            'term': term
        })
    
    except (json.JSONDecodeError, KeyError):
        return JsonResponse({'error': 'Dados inválidos'}, status=400)


@csrf_exempt
@require_http_methods(["POST"])
def search_analytics(request):
    """
    View para registrar analytics de busca
    """
    try:
        data = json.loads(request.body)
        
        # Registrar busca para analytics
        analytics = SearchAnalytics()
        analytics.track_search(
            filters=data.get('filters'),
            results_count=data.get('results_count', 0),
            user_id=request.user.id if request.user.is_authenticated else None
        )
        
        return JsonResponse({'status': 'success'})
    
    except (json.JSONDecodeError, KeyError):
        return JsonResponse({'error': 'Dados inválidos'}, status=400)


@require_http_methods(["GET"])
def get_cities_by_state(request, estado):
    """
    View AJAX para obter cidades por estado
    """
    if not estado:
        return JsonResponse({'cities': []})
    
    from models.models import Modelo
    
    cidades = Modelo.objects.filter(
        estado=estado,
        status='ativo'
    ).values_list('cidade', flat=True).distinct().order_by('cidade')
    
    return JsonResponse({
        'cities': list(cidades)
    })


@require_http_methods(["GET"])
def search_trends(request):
    """
    View para obter tendências de busca
    """
    analytics = SearchAnalytics()
    trends = analytics.get_search_trends(days=7)
    
    return JsonResponse(trends)


@require_http_methods(["GET"])
def popular_searches(request):
    """
    View para obter buscas populares
    """
    suggestion_engine = SearchSuggestionEngine()
    popular = suggestion_engine.get_popular_searches(limit=10)
    
    return JsonResponse({
        'popular_searches': popular
    })


def search_results_export(request):
    """
    View para exportar resultados de busca
    """
    # Implementar exportação (CSV, Excel, etc.)
    pass


# Funções auxiliares
def _convert_quick_to_advanced(quick_data):
    """Converte dados de busca rápida para busca avançada"""
    from .engine import SearchFilters
    
    # Extrair termo de busca
    termo_busca = quick_data.get('q', '')
    localidade = quick_data.get('localidade', '')
    categoria = quick_data.get('categoria', '')
    
    # Determinar se localidade é cidade ou estado
    estado = None
    cidade = None
    
    if localidade:
        # Lista de estados brasileiros
        estados = [
            'AC', 'AL', 'AP', 'AM', 'BA', 'CE', 'DF', 'ES', 'GO', 'MA', 'MT', 'MS',
            'MG', 'PA', 'PB', 'PR', 'PE', 'PI', 'RJ', 'RN', 'RS', 'RO', 'RR', 'SC',
            'SP', 'SE', 'TO'
        ]
        
        if localidade.upper() in estados:
            estado = localidade.upper()
        else:
            cidade = localidade
    
    # Combinar termos de busca
    if termo_busca and localidade:
        termo_completo = f"{termo_busca} {localidade}"
    else:
        termo_completo = termo_busca or localidade
    
    return SearchFilters(
        termo_busca=termo_completo,
        estado=estado,
        cidade=cidade,
        categoria_servico=[categoria] if categoria else None,
        ordenacao='recentes',
        por_pagina=20
    )


def _get_estados_disponiveis():
    """Retorna lista de estados com modelos ativos"""
    from models.models import Modelo
    
    return Modelo.objects.filter(
        status='ativo'
    ).values_list('estado', flat=True).distinct().order_by('estado')


def _get_cidades_populares():
    """Retorna lista de cidades populares"""
    from models.models import Modelo
    
    return Modelo.objects.filter(
        status='ativo'
    ).values('cidade', 'estado').annotate(
        count=Count('id')
    ).order_by('-count')[:20]


def _get_categorias_disponiveis():
    """Retorna lista de categorias disponíveis"""
    from models.models import Categoria
    
    return Categoria.objects.filter(ativo=True).order_by('ordem', 'nome')


# View base para busca com funcionalidades comuns
class BaseSearchView(View):
    """View base para funcionalidades de busca"""
    
    def get_search_engine(self):
        """Retorna instância do motor de busca"""
        return AdvancedSearchEngine()
    
    def get_suggestion_engine(self):
        """Retorna instância do motor de sugestões"""
        return SearchSuggestionEngine()
    
    def get_analytics(self):
        """Retorna instância do analytics"""
        return SearchAnalytics()
    
    def apply_common_context(self, context):
        """Aplica contexto comum a todas as views de busca"""
        context.update({
            'estados_disponiveis': _get_estados_disponiveis(),
            'cidades_populares': _get_cidades_populares(),
            'categorias_disponiveis': _get_categorias_disponiveis(),
        })
        return context 