"""
Formulários para busca avançada de modelos
"""

from django import forms
from django.core.exceptions import ValidationError
from models.models import Modelo, Categoria


class AdvancedSearchForm(forms.Form):
    """Formulário principal de busca avançada"""
    
    # Busca por texto
    termo_busca = forms.CharField(
        max_length=200,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Buscar por nome, cidade, descrição...',
            'autocomplete': 'off'
        }),
        help_text='Digite palavras-chave para buscar'
    )
    
    # Filtros básicos
    categoria_servico = forms.MultipleChoiceField(
        choices=[
            ('JOB', 'Anúncios'),
            ('FAB', 'Venda de Conteúdos'),
            ('CAM', 'Câmera Privativa'),
        ],
        required=False,
        widget=forms.CheckboxSelectMultiple(attrs={
            'class': 'form-check-input'
        }),
        help_text='Tipo de serviço oferecido'
    )
    
    tipo_ensaio = forms.MultipleChoiceField(
        choices=Modelo.TIPO_ENSAIO_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple(attrs={
            'class': 'form-check-input'
        }),
        help_text='Tipo de anúncio'
    )
    
    etnia = forms.MultipleChoiceField(
        choices=Modelo.ETNIA_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple(attrs={
            'class': 'form-check-input'
        }),
        help_text='Etnia'
    )
    
    local_atendimento = forms.MultipleChoiceField(
        choices=Modelo.LOCAL_ATENDIMENTO_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple(attrs={
            'class': 'form-check-input'
        }),
        help_text='Local de atendimento'
    )
    
    # Localização
    estado = forms.ChoiceField(
        choices=[('', 'Selecione um estado')] + [
            ('AC', 'Acre'), ('AL', 'Alagoas'), ('AP', 'Amapá'), ('AM', 'Amazonas'),
            ('BA', 'Bahia'), ('CE', 'Ceará'), ('DF', 'Distrito Federal'), ('ES', 'Espírito Santo'),
            ('GO', 'Goiás'), ('MA', 'Maranhão'), ('MT', 'Mato Grosso'), ('MS', 'Mato Grosso do Sul'),
            ('MG', 'Minas Gerais'), ('PA', 'Pará'), ('PB', 'Paraíba'), ('PR', 'Paraná'),
            ('PE', 'Pernambuco'), ('PI', 'Piauí'), ('RJ', 'Rio de Janeiro'), ('RN', 'Rio Grande do Norte'),
            ('RS', 'Rio Grande do Sul'), ('RO', 'Rondônia'), ('RR', 'Roraima'), ('SC', 'Santa Catarina'),
            ('SP', 'São Paulo'), ('SE', 'Sergipe'), ('TO', 'Tocantins')
        ],
        required=False,
        widget=forms.Select(attrs={
            'class': 'form-select',
            'id': 'estado-select'
        })
    )
    
    cidade = forms.CharField(
        max_length=100,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Digite a cidade...',
            'id': 'cidade-input'
        })
    )
    
    # Filtros de idade
    idade_min = forms.IntegerField(
        min_value=18,
        max_value=99,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '18',
            'min': '18',
            'max': '99'
        }),
        help_text='Idade mínima'
    )
    
    idade_max = forms.IntegerField(
        min_value=18,
        max_value=99,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '99',
            'min': '18',
            'max': '99'
        }),
        help_text='Idade máxima'
    )
    
    # Filtros de preço
    preco_min = forms.DecimalField(
        max_digits=10,
        decimal_places=2,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '0.00',
            'step': '0.01',
            'min': '0'
        }),
        help_text='Preço mínimo'
    )
    
    preco_max = forms.DecimalField(
        max_digits=10,
        decimal_places=2,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '1000.00',
            'step': '0.01',
            'min': '0'
        }),
        help_text='Preço máximo'
    )
    
    faixas_preco = forms.MultipleChoiceField(
        choices=[
            ('ate_100', 'Até R$ 100'),
            ('100_200', 'R$ 100 - R$ 200'),
            ('200_300', 'R$ 200 - R$ 300'),
            ('acima_300', 'Acima de R$ 300'),
        ],
        required=False,
        widget=forms.CheckboxSelectMultiple(attrs={
            'class': 'form-check-input'
        }),
        help_text='Faixas de preço'
    )
    
    # Filtros de características físicas
    altura_min = forms.IntegerField(
        min_value=140,
        max_value=200,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '140',
            'min': '140',
            'max': '200'
        }),
        help_text='Altura mínima (cm)'
    )
    
    altura_max = forms.IntegerField(
        min_value=140,
        max_value=200,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '200',
            'min': '140',
            'max': '200'
        }),
        help_text='Altura máxima (cm)'
    )
    
    peso_min = forms.IntegerField(
        min_value=40,
        max_value=150,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '40',
            'min': '40',
            'max': '150'
        }),
        help_text='Peso mínimo (kg)'
    )
    
    peso_max = forms.IntegerField(
        min_value=40,
        max_value=150,
        required=False,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': '150',
            'min': '40',
            'max': '150'
        }),
        help_text='Peso máximo (kg)'
    )
    
    # Filtros de avaliação
    nota_min = forms.ChoiceField(
        choices=[
            ('', 'Qualquer nota'),
            ('4.5', '4.5+ estrelas'),
            ('4.0', '4.0+ estrelas'),
            ('3.5', '3.5+ estrelas'),
            ('3.0', '3.0+ estrelas'),
        ],
        required=False,
        widget=forms.Select(attrs={
            'class': 'form-select'
        }),
        help_text='Nota mínima'
    )
    
    com_avaliacoes = forms.BooleanField(
        required=False,
        widget=forms.CheckboxInput(attrs={
            'class': 'form-check-input'
        }),
        help_text='Apenas modelos com avaliações'
    )
    
    # Filtros de disponibilidade
    disponivel_agora = forms.BooleanField(
        required=False,
        widget=forms.CheckboxInput(attrs={
            'class': 'form-check-input'
        }),
        help_text='Apenas modelos online agora'
    )
    
    # Filtros de status
    premium = forms.BooleanField(
        required=False,
        widget=forms.CheckboxInput(attrs={
            'class': 'form-check-input'
        }),
        help_text='Apenas modelos premium'
    )
    
    boost_ativo = forms.BooleanField(
        required=False,
        widget=forms.CheckboxInput(attrs={
            'class': 'form-check-input'
        }),
        help_text='Apenas modelos com boost ativo'
    )
    
    online = forms.BooleanField(
        required=False,
        widget=forms.CheckboxInput(attrs={
            'class': 'form-check-input'
        }),
        help_text='Apenas modelos online'
    )
    
    # Ordenação
    ordenacao = forms.ChoiceField(
        choices=[
            ('recentes', 'Mais Recentes'),
            ('mais_vistas_total', 'Mais Vistas (Total)'),
            ('mais_vistas_mes', 'Mais Vistas (Mês)'),
            ('mais_favoritadas', 'Mais Favoritadas'),
            ('melhor_avaliadas', 'Melhor Avaliadas'),
            ('preco_menor', 'Menor Preço'),
            ('preco_maior', 'Maior Preço'),
            ('online', 'Online Primeiro'),
        ],
        required=False,
        initial='recentes',
        widget=forms.Select(attrs={
            'class': 'form-select'
        })
    )
    
    # Paginação
    por_pagina = forms.ChoiceField(
        choices=[
            ('12', '12 por página'),
            ('20', '20 por página'),
            ('40', '40 por página'),
            ('60', '60 por página'),
        ],
        required=False,
        initial='20',
        widget=forms.Select(attrs={
            'class': 'form-select'
        })
    )
    
    def clean(self):
        """Validação personalizada do formulário"""
        cleaned_data = super().clean()
        
        # Validar idade
        idade_min = cleaned_data.get('idade_min')
        idade_max = cleaned_data.get('idade_max')
        if idade_min and idade_max and idade_min > idade_max:
            raise ValidationError("Idade mínima não pode ser maior que a máxima.")
        
        # Validar preço
        preco_min = cleaned_data.get('preco_min')
        preco_max = cleaned_data.get('preco_max')
        if preco_min and preco_max and preco_min > preco_max:
            raise ValidationError("Preço mínimo não pode ser maior que o máximo.")
        
        # Validar altura
        altura_min = cleaned_data.get('altura_min')
        altura_max = cleaned_data.get('altura_max')
        if altura_min and altura_max and altura_min > altura_max:
            raise ValidationError("Altura mínima não pode ser maior que a máxima.")
        
        # Validar peso
        peso_min = cleaned_data.get('peso_min')
        peso_max = cleaned_data.get('peso_max')
        if peso_min and peso_max and peso_min > peso_max:
            raise ValidationError("Peso mínimo não pode ser maior que o máximo.")
        
        return cleaned_data
    
    def get_search_filters(self):
        """Converte o formulário para objeto SearchFilters"""
        from .engine import SearchFilters
        
        return SearchFilters(
            categoria_servico=self.cleaned_data.get('categoria_servico'),
            tipo_ensaio=self.cleaned_data.get('tipo_ensaio'),
            estado=self.cleaned_data.get('estado'),
            cidade=self.cleaned_data.get('cidade'),
            local_atendimento=self.cleaned_data.get('local_atendimento'),
            etnia=self.cleaned_data.get('etnia'),
            idade_min=self.cleaned_data.get('idade_min'),
            idade_max=self.cleaned_data.get('idade_max'),
            preco_min=self.cleaned_data.get('preco_min'),
            preco_max=self.cleaned_data.get('preco_max'),
            faixas_preco=self.cleaned_data.get('faixas_preco'),
            disponivel_agora=self.cleaned_data.get('disponivel_agora'),
            altura_min=self.cleaned_data.get('altura_min'),
            altura_max=self.cleaned_data.get('altura_max'),
            peso_min=self.cleaned_data.get('peso_min'),
            peso_max=self.cleaned_data.get('peso_max'),
            nota_min=float(self.cleaned_data.get('nota_min')) if self.cleaned_data.get('nota_min') else None,
            com_avaliacoes=self.cleaned_data.get('com_avaliacoes'),
            premium=self.cleaned_data.get('premium'),
            boost_ativo=self.cleaned_data.get('boost_ativo'),
            online=self.cleaned_data.get('online'),
            termo_busca=self.cleaned_data.get('termo_busca'),
            ordenacao=self.cleaned_data.get('ordenacao'),
            por_pagina=int(self.cleaned_data.get('por_pagina') or '20')
        )


class QuickSearchForm(forms.Form):
    """Formulário de busca rápida"""
    
    q = forms.CharField(
        max_length=200,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Buscar modelos...',
            'autocomplete': 'off'
        })
    )
    
    localidade = forms.CharField(
        max_length=100,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Cidade ou Estado...'
        })
    )
    
    categoria = forms.ChoiceField(
        choices=[('', 'Todas as categorias')] + [
            ('JOB', 'Anúncios'),
            ('FAB', 'Venda de Conteúdos'),
            ('CAM', 'Câmera Privativa'),
        ],
        required=False,
        widget=forms.Select(attrs={
            'class': 'form-select'
        })
    )


class SearchSuggestionForm(forms.Form):
    """Formulário para sugestões de busca"""
    
    term = forms.CharField(
        max_length=100,
        required=True,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Digite para buscar...',
            'autocomplete': 'off'
        })
    ) 