"""
Motor de busca avançada para modelos
Implementa múltiplas estratégias de busca e filtros avançados
"""

from django.db.models import Q, Count, Case, When, IntegerField, Avg, F, Value, Min, Max
from django.db.models.functions import Coalesce, ExtractHour, ExtractDay
from django.utils import timezone
from django.contrib.postgres.search import SearchVector, SearchQuery, SearchRank
from datetime import timedelta
import re
from typing import Dict, List, Any, Optional, Tuple
from dataclasses import dataclass


@dataclass
class SearchFilters:
    """Classe para armazenar filtros de busca"""
    # Filtros básicos
    categoria_servico: List[str] = None
    tipo_ensaio: List[str] = None
    estado: str = None
    cidade: str = None
    local_atendimento: List[str] = None
    etnia: List[str] = None
    
    # Filtros de idade
    idade_min: int = None
    idade_max: int = None
    
    # Filtros de preço
    preco_min: float = None
    preco_max: float = None
    faixas_preco: List[str] = None
    
    # Filtros de disponibilidade
    disponivel_agora: bool = None
    horario_inicio: int = None
    horario_fim: int = None
    dias_semana: List[int] = None
    
    # Filtros de características físicas
    altura_min: int = None
    altura_max: int = None
    peso_min: int = None
    peso_max: int = None
    
    # Filtros de avaliação
    nota_min: float = None
    com_avaliacoes: bool = None
    
    # Filtros de popularidade
    mais_vistas: bool = None
    mais_favoritadas: bool = None
    
    # Filtros de status
    premium: bool = None
    boost_ativo: bool = None
    online: bool = None
    
    # Busca por texto
    termo_busca: str = None
    
    # Ordenação
    ordenacao: str = 'recentes'
    
    # Paginação
    pagina: int = 1
    por_pagina: int = 20


class AdvancedSearchEngine:
    """Motor de busca avançada para modelos"""
    
    def __init__(self, queryset=None):
        from models.models import Modelo
        self.base_queryset = queryset or Modelo.objects.filter(
            status='ativo'
        )
    
    def search(self, filters: SearchFilters) -> Tuple[Any, Dict[str, Any]]:
        """
        Executa a busca avançada com todos os filtros aplicados
        
        Args:
            filters: Objeto SearchFilters com todos os filtros
            
        Returns:
            Tuple com queryset filtrado e estatísticas da busca
        """
        queryset = self.base_queryset
        
        # Aplicar anotações para métricas
        queryset = self._apply_annotations(queryset)
        
        # Aplicar filtros básicos
        queryset = self._apply_basic_filters(queryset, filters)
        
        # Aplicar filtros de texto
        queryset = self._apply_text_search(queryset, filters)
        
        # Aplicar filtros de preço
        queryset = self._apply_price_filters(queryset, filters)
        
        # Aplicar filtros de disponibilidade
        queryset = self._apply_availability_filters(queryset, filters)
        
        # Aplicar filtros de características físicas
        queryset = self._apply_physical_filters(queryset, filters)
        
        # Aplicar filtros de avaliação
        queryset = self._apply_rating_filters(queryset, filters)
        
        # Aplicar filtros de popularidade
        queryset = self._apply_popularity_filters(queryset, filters)
        
        # Aplicar filtros de status
        queryset = self._apply_status_filters(queryset, filters)
        
        # Aplicar ordenação
        queryset = self._apply_ordering(queryset, filters)
        
        # Calcular estatísticas
        stats = self._calculate_search_stats(queryset, filters)
        
        return queryset, stats
    
    def _apply_annotations(self, queryset):
        """Aplica anotações para métricas e cálculos"""
        return queryset.annotate(
            # Visualizações
            total_views=Count('visualizacoes'),
            monthly_views=Count(
                Case(
                    When(
                        visualizacoes__data_visualizacao__gte=timezone.now() - timedelta(days=30),
                        then=1
                    ),
                    output_field=IntegerField()
                )
            ),
            # Favoritos
            total_favorites=Count('favoritado_por'),
            # Avaliações
            avg_rating=Coalesce(Avg('avaliacoes__nota'), Value(0.0)),
            total_ratings=Count('avaliacoes'),
            # Boost priority (agora baseado em plano ativo)
            boost_priority=Case(
                When(
                    plano_ativo__isnull=False,
                    plano_data_fim__gt=timezone.now(),
                    then=Value(0)
                ),
                default=Value(1),
                output_field=IntegerField()
            ),
            # Online status (simulado - pode ser implementado com websockets)
            is_online=Case(
                When(
                    ultima_atividade__gte=timezone.now() - timedelta(minutes=30),
                    then=Value(1)
                ),
                default=Value(0),
                output_field=IntegerField()
            )
        )
    
    def _apply_basic_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros básicos"""
        if filters.categoria_servico:
            queryset = queryset.filter(categoria_servico__in=filters.categoria_servico)
        
        if filters.tipo_ensaio:
            queryset = queryset.filter(tipo_ensaio__in=filters.tipo_ensaio)
        
        if filters.estado:
            queryset = queryset.filter(estado=filters.estado)
        
        if filters.cidade:
            queryset = queryset.filter(cidade__icontains=filters.cidade)
        
        if filters.local_atendimento:
            queryset = queryset.filter(local_atendimento__in=filters.local_atendimento)
        
        if filters.etnia:
            queryset = queryset.filter(etnia__in=filters.etnia)
        
        if filters.idade_min:
            queryset = queryset.filter(idade__gte=filters.idade_min)
        
        if filters.idade_max:
            queryset = queryset.filter(idade__lte=filters.idade_max)
        
        return queryset
    
    def _apply_text_search(self, queryset, filters: SearchFilters):
        """Aplica busca por texto usando múltiplas estratégias"""
        if not filters.termo_busca:
            return queryset
        
        termo = filters.termo_busca.strip()
        if not termo:
            return queryset
        
        # Busca em múltiplos campos
        search_conditions = Q()
        
        # Busca exata
        search_conditions |= Q(nome_exibicao__iexact=termo)
        search_conditions |= Q(cidade__iexact=termo)
        
        # Busca parcial
        search_conditions |= Q(nome_exibicao__icontains=termo)
        search_conditions |= Q(sobre_mim__icontains=termo)
        search_conditions |= Q(cidade__icontains=termo)
        
        # Busca por palavras-chave
        palavras = termo.split()
        for palavra in palavras:
            if len(palavra) > 2:  # Ignorar palavras muito curtas
                search_conditions |= Q(nome_exibicao__icontains=palavra)
                search_conditions |= Q(sobre_mim__icontains=palavra)
                search_conditions |= Q(cidade__icontains=palavra)
        
        # Busca por categorias
        search_conditions |= Q(categorias__nome__icontains=termo)
        
        # Busca por serviços
        search_conditions |= Q(servicos__descricao__icontains=termo)
        
        return queryset.filter(search_conditions).distinct()
    
    def _apply_price_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros de preço"""
        if filters.preco_min is not None:
            queryset = queryset.filter(servicos__preco__gte=filters.preco_min)
        
        if filters.preco_max is not None:
            queryset = queryset.filter(servicos__preco__lte=filters.preco_max)
        
        if filters.faixas_preco:
            price_conditions = Q()
            for faixa in filters.faixas_preco:
                if faixa == 'ate_100':
                    price_conditions |= Q(servicos__preco__lte=100)
                elif faixa == '100_200':
                    price_conditions |= Q(servicos__preco__gte=100, servicos__preco__lte=200)
                elif faixa == '200_300':
                    price_conditions |= Q(servicos__preco__gte=200, servicos__preco__lte=300)
                elif faixa == 'acima_300':
                    price_conditions |= Q(servicos__preco__gt=300)
            
            if price_conditions:
                queryset = queryset.filter(price_conditions).distinct()
        
        return queryset
    
    def _apply_availability_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros de disponibilidade"""
        if filters.disponivel_agora:
            # Modelos online nas últimas 30 minutos
            queryset = queryset.filter(
                ultima_atividade__gte=timezone.now() - timedelta(minutes=30)
            )
        
        if filters.horario_inicio is not None and filters.horario_fim is not None:
            # Filtro por horário (pode ser expandido com tabela de disponibilidade)
            pass
        
        if filters.dias_semana:
            # Filtro por dias da semana (pode ser expandido)
            pass
        
        return queryset
    
    def _apply_physical_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros de características físicas"""
        if filters.altura_min:
            queryset = queryset.filter(altura__gte=filters.altura_min)
        
        if filters.altura_max:
            queryset = queryset.filter(altura__lte=filters.altura_max)
        
        if filters.peso_min:
            queryset = queryset.filter(peso__gte=filters.peso_min)
        
        if filters.peso_max:
            queryset = queryset.filter(peso__lte=filters.peso_max)
        
        return queryset
    
    def _apply_rating_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros de avaliação"""
        if filters.nota_min:
            queryset = queryset.filter(avg_rating__gte=filters.nota_min)
        
        if filters.com_avaliacoes:
            queryset = queryset.filter(total_ratings__gt=0)
        
        return queryset
    
    def _apply_popularity_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros de popularidade"""
        if filters.mais_vistas:
            queryset = queryset.filter(total_views__gt=0)
        
        if filters.mais_favoritadas:
            queryset = queryset.filter(total_favorites__gt=0)
        
        return queryset
    
    def _apply_status_filters(self, queryset, filters: SearchFilters):
        """Aplica filtros de status"""
        if filters.premium is not None:
            # Premium agora é baseado em plano ativo
            if filters.premium:
                queryset = queryset.filter(
                    plano_ativo__isnull=False,
                    plano_data_fim__gt=timezone.now()
                )
            else:
                queryset = queryset.filter(
                    Q(plano_ativo__isnull=True) | Q(plano_data_fim__lte=timezone.now())
                )
        
        if filters.boost_ativo is not None:
            if filters.boost_ativo:
                queryset = queryset.filter(
                    plano_ativo__isnull=False,
                    plano_data_fim__gt=timezone.now()
                )
            else:
                queryset = queryset.filter(
                    Q(plano_ativo__isnull=True) | Q(plano_data_fim__lte=timezone.now())
                )
        
        if filters.online is not None:
            if filters.online:
                queryset = queryset.filter(
                    ultima_atividade__gte=timezone.now() - timedelta(minutes=30)
                )
            else:
                queryset = queryset.filter(
                    ultima_atividade__lt=timezone.now() - timedelta(minutes=30)
                )
        
        return queryset
    
    def _apply_ordering(self, queryset, filters: SearchFilters):
        """Aplica ordenação"""
        if filters.ordenacao == 'mais_vistas_total':
            queryset = queryset.order_by('-total_views', 'boost_priority', '-data_cadastro')
        elif filters.ordenacao == 'mais_vistas_mes':
            queryset = queryset.order_by('-monthly_views', 'boost_priority', '-data_cadastro')
        elif filters.ordenacao == 'mais_favoritadas':
            queryset = queryset.order_by('-total_favorites', 'boost_priority', '-data_cadastro')
        elif filters.ordenacao == 'melhor_avaliadas':
            queryset = queryset.order_by('-avg_rating', '-total_ratings', 'boost_priority', '-data_cadastro')
        elif filters.ordenacao == 'preco_menor':
            queryset = queryset.order_by('servicos__preco', 'boost_priority', '-data_cadastro')
        elif filters.ordenacao == 'preco_maior':
            queryset = queryset.order_by('-servicos__preco', 'boost_priority', '-data_cadastro')
        elif filters.ordenacao == 'online':
            queryset = queryset.order_by('-is_online', 'boost_priority', '-data_cadastro')
        else:  # recentes
            queryset = queryset.order_by('boost_priority', '-data_cadastro')
        
        return queryset
    
    def _calculate_search_stats(self, queryset, filters: SearchFilters) -> Dict[str, Any]:
        """Calcula estatísticas da busca"""
        total_results = queryset.count()
        
        # Estatísticas por categoria
        stats_by_category = queryset.values('categoria_servico').annotate(
            count=Count('id')
        ).order_by('-count')
        
        # Estatísticas por estado
        stats_by_state = queryset.values('estado').annotate(
            count=Count('id')
        ).order_by('-count')
        
        # Estatísticas por etnia
        stats_by_ethnicity = queryset.values('etnia').annotate(
            count=Count('id')
        ).order_by('-count')
        
        # Faixa de preços - simplificado para evitar problemas de tipo
        try:
            price_stats = queryset.aggregate(
                min_price=Min('servicos__preco'),
                max_price=Max('servicos__preco'),
                avg_price=Avg('servicos__preco')
            )
            # Converter None para 0
            price_stats = {
                'min_price': price_stats['min_price'] or 0.0,
                'max_price': price_stats['max_price'] or 0.0,
                'avg_price': price_stats['avg_price'] or 0.0
            }
        except:
            price_stats = {
                'min_price': 0.0,
                'max_price': 0.0,
                'avg_price': 0.0
            }
        
        # Faixa de idades
        age_stats = queryset.aggregate(
            min_age=Min('idade'),
            max_age=Max('idade'),
            avg_age=Avg('idade')
        )
        
        return {
            'total_results': total_results,
            'stats_by_category': list(stats_by_category),
            'stats_by_state': list(stats_by_state),
            'stats_by_ethnicity': list(stats_by_ethnicity),
            'price_stats': price_stats,
            'age_stats': age_stats,
            'filters_applied': self._get_applied_filters_summary(filters)
        }
    
    def _get_applied_filters_summary(self, filters: SearchFilters) -> Dict[str, Any]:
        """Retorna um resumo dos filtros aplicados"""
        applied = {}
        
        if filters.categoria_servico:
            applied['categoria_servico'] = filters.categoria_servico
        if filters.tipo_ensaio:
            applied['tipo_ensaio'] = filters.tipo_ensaio
        if filters.estado:
            applied['estado'] = filters.estado
        if filters.cidade:
            applied['cidade'] = filters.cidade
        if filters.etnia:
            applied['etnia'] = filters.etnia
        if filters.idade_min or filters.idade_max:
            applied['idade'] = f"{filters.idade_min or 18}-{filters.idade_max or 99}"
        if filters.preco_min or filters.preco_max:
            applied['preco'] = f"R$ {filters.preco_min or 0}-{filters.preco_max or '∞'}"
        if filters.termo_busca:
            applied['busca'] = filters.termo_busca
        
        return applied


class SearchSuggestionEngine:
    """Motor de sugestões de busca"""
    
    def __init__(self):
        from models.models import Modelo, Categoria
        
    def get_suggestions(self, term: str, limit: int = 5) -> List[str]:
        """Retorna sugestões de busca baseadas no termo"""
        suggestions = []
        
        # Verificar se o termo é válido
        if not term or len(term.strip()) < 2:
            return []
        
        term = term.strip()
        
        # Buscar nomes de modelos
        from models.models import Modelo
        modelos = Modelo.objects.filter(
            nome_exibicao__icontains=term,
            status='ativo'
        ).values_list('nome_exibicao', flat=True)[:limit]
        suggestions.extend(modelos)
        
        # Buscar cidades
        cidades = Modelo.objects.filter(
            cidade__icontains=term,
            status='ativo'
        ).values_list('cidade', flat=True).distinct()[:limit]
        suggestions.extend(cidades)
        
        # Buscar categorias
        from models.models import Categoria
        categorias = Categoria.objects.filter(
            nome__icontains=term,
            ativo=True
        ).values_list('nome', flat=True)[:limit]
        suggestions.extend(categorias)
        
        return list(set(suggestions))[:limit]
    
    def get_popular_searches(self, limit: int = 10) -> List[str]:
        """Retorna buscas populares baseadas em histórico"""
        # Implementar com cache Redis ou similar
        return [
            "São Paulo", "Rio de Janeiro", "Belo Horizonte",
            "Premium", "Online", "Mais Vistas"
        ][:limit]


class SearchAnalytics:
    """Analytics para o sistema de busca"""
    
    def __init__(self):
        pass
    
    def track_search(self, filters: SearchFilters, results_count: int, user_id: int = None):
        """Registra uma busca para analytics"""
        # Implementar com cache Redis ou banco de dados
        pass
    
    def get_search_trends(self, days: int = 7) -> Dict[str, Any]:
        """Retorna tendências de busca"""
        # Implementar analytics
        return {
            'popular_terms': [],
            'popular_filters': [],
            'search_volume': 0
        } 