#!/bin/bash

# Script de deploy para o projeto Allure
# Este script automatiza o processo de deploy em produção

set -e

echo "🚀 Iniciando deploy do projeto Allure..."

# Cores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Função para log colorido
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Verificar se estamos no diretório correto
if [ ! -f "manage.py" ]; then
    log_error "Este script deve ser executado no diretório raiz do projeto Django"
    exit 1
fi

# Verificar se o Docker está instalado
if ! command -v docker &> /dev/null; then
    log_error "Docker não está instalado. Por favor, instale o Docker primeiro."
    exit 1
fi

# Verificar se o Docker Compose está instalado
if ! command -v docker-compose &> /dev/null; then
    log_error "Docker Compose não está instalado. Por favor, instale o Docker Compose primeiro."
    exit 1
fi

# Verificar se existe arquivo .env
if [ ! -f .env ]; then
    log_warn "Arquivo .env não encontrado. Criando a partir do exemplo..."
    cp env.example .env
    log_error "Por favor, configure o arquivo .env antes de continuar o deploy."
    exit 1
fi

# Função para backup do banco
backup_database() {
    log_info "Fazendo backup do banco de dados..."
    mkdir -p backups
    timestamp=$(date +"%Y%m%d_%H%M%S")
    backup_file="backups/db_backup_${timestamp}.sql"
    
    if docker-compose ps db | grep -q "Up"; then
        docker-compose exec -T db pg_dump -U allure_user allure_db > "$backup_file"
        log_info "Backup salvo em: $backup_file"
    else
        log_warn "Banco de dados não está rodando. Pulando backup."
    fi
}

# Função para deploy
deploy() {
    log_info "Iniciando processo de deploy..."
    
    # 1. Parar containers existentes
    log_info "Parando containers existentes..."
    docker-compose down || true
    
    # 2. Fazer backup do banco (se existir)
    backup_database
    
    # 3. Construir novas imagens
    log_info "Construindo imagens Docker..."
    docker-compose build --no-cache
    
    # 4. Iniciar serviços
    log_info "Iniciando serviços..."
    docker-compose up -d
    
    # 5. Aguardar banco estar pronto
    log_info "Aguardando banco de dados estar pronto..."
    sleep 10
    
    # 6. Executar migrações
    log_info "Executando migrações do banco de dados..."
    docker-compose exec web python manage.py migrate
    
    # 7. Coletar arquivos estáticos
    log_info "Coletando arquivos estáticos..."
    docker-compose exec web python manage.py collectstatic --noinput
    
    # 8. Criar superusuário (se não existir)
    log_info "Verificando superusuário..."
    docker-compose exec web python manage.py shell -c "
from django.contrib.auth import get_user_model
User = get_user_model()
if not User.objects.filter(is_superuser=True).exists():
    User.objects.create_superuser('admin', 'admin@allure.com.br', 'admin123')
    print('Superusuário criado: admin/admin123')
else:
    print('Superusuário já existe')
"
    
    log_info "Deploy concluído com sucesso!"
    log_info "Aplicação disponível em: http://localhost:8000"
    log_info "Admin disponível em: http://localhost:8000/admin"
}

# Função para rollback
rollback() {
    log_warn "Iniciando rollback..."
    
    # Parar containers
    docker-compose down
    
    # Restaurar backup mais recente
    latest_backup=$(ls -t backups/db_backup_*.sql 2>/dev/null | head -n1)
    if [ -n "$latest_backup" ]; then
        log_info "Restaurando backup: $latest_backup"
        docker-compose up -d db
        sleep 10
        docker-compose exec -T db psql -U allure_user -d allure_db < "$latest_backup"
        log_info "Rollback concluído"
    else
        log_error "Nenhum backup encontrado para rollback"
        exit 1
    fi
}

# Função para mostrar status
show_status() {
    log_info "Status dos containers:"
    docker-compose ps
    
    log_info "Uso de recursos:"
    docker stats --no-stream
}

# Função para mostrar logs
show_logs() {
    log_info "Mostrando logs dos containers:"
    docker-compose logs -f
}

# Função para mostrar ajuda
show_help() {
    echo "Uso: $0 [comando]"
    echo ""
    echo "Comandos disponíveis:"
    echo "  deploy   - Executar deploy completo"
    echo "  rollback - Fazer rollback para versão anterior"
    echo "  status   - Mostrar status dos containers"
    echo "  logs     - Mostrar logs dos containers"
    echo "  backup   - Fazer backup do banco de dados"
    echo "  help     - Mostrar esta ajuda"
}

# Processar argumentos
case "${1:-deploy}" in
    deploy)
        deploy
        ;;
    rollback)
        rollback
        ;;
    status)
        show_status
        ;;
    logs)
        show_logs
        ;;
    backup)
        backup_database
        ;;
    help|*)
        show_help
        ;;
esac
