#!/usr/bin/env python
"""
Script de teste principal para o projeto Allure
Executa todos os testes e gera um relatório detalhado
"""

import os
import sys
import subprocess
import json
import time
from datetime import datetime
from pathlib import Path

# Adicionar o diretório do projeto ao path
BASE_DIR = Path(__file__).resolve().parent
sys.path.insert(0, str(BASE_DIR))

# Configurações do Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')

import django
django.setup()

from django.test.utils import get_runner
from django.conf import settings


class TestRunner:
    def __init__(self):
        self.results = {
            'timestamp': datetime.now().isoformat(),
            'total_tests': 0,
            'passed': 0,
            'failed': 0,
            'errors': 0,
            'skipped': 0,
            'test_details': [],
            'issues_found': [],
            'recommendations': []
        }
        
    def run_all_tests(self):
        """Executa todos os testes do projeto"""
        print("🚀 Iniciando execução de testes do projeto Allure...")
        print("=" * 60)
        
        # Configurar o test runner do Django
        TestRunner = get_runner(settings)
        test_runner = TestRunner()
        
        # Lista de apps para testar
        apps_to_test = ['core', 'accounts', 'models']
        
        for app in apps_to_test:
            print(f"\n📋 Testando app: {app}")
            print("-" * 40)
            
            try:
                # Executar testes do app
                start_time = time.time()
                result = test_runner.run_tests([app])
                end_time = time.time()
                
                # Processar resultados
                self.process_test_results(app, result, end_time - start_time)
                
            except Exception as e:
                print(f"❌ Erro ao testar {app}: {str(e)}")
                self.results['errors'] += 1
                self.results['issues_found'].append({
                    'app': app,
                    'type': 'test_execution_error',
                    'message': str(e)
                })
        
        # Executar testes de integração
        self.run_integration_tests()
        
        # Gerar relatório
        self.generate_report()
        
    def process_test_results(self, app, result, duration):
        """Processa os resultados dos testes"""
        if hasattr(result, 'failures'):
            failures = result.failures
        else:
            failures = []
            
        if hasattr(result, 'errors'):
            errors = result.errors
        else:
            errors = []
            
        if hasattr(result, 'skipped'):
            skipped = result.skipped
        else:
            skipped = []
            
        # Contar testes
        total_tests = len(failures) + len(errors) + len(skipped)
        if hasattr(result, 'testsRun'):
            total_tests = result.testsRun
            
        passed = total_tests - len(failures) - len(errors) - len(skipped)
        
        # Atualizar contadores globais
        self.results['total_tests'] += total_tests
        self.results['passed'] += passed
        self.results['failed'] += len(failures)
        self.results['errors'] += len(errors)
        self.results['skipped'] += len(skipped)
        
        # Adicionar detalhes do app
        app_result = {
            'app': app,
            'total': total_tests,
            'passed': passed,
            'failed': len(failures),
            'errors': len(errors),
            'skipped': len(skipped),
            'duration': round(duration, 2),
            'failures': [str(f) for f in failures],
            'errors': [str(e) for e in errors]
        }
        
        self.results['test_details'].append(app_result)
        
        # Exibir resultados do app
        print(f"✅ Passaram: {passed}")
        print(f"❌ Falharam: {len(failures)}")
        print(f"⚠️  Erros: {len(errors)}")
        print(f"⏭️  Pulados: {len(skipped)}")
        print(f"⏱️  Duração: {duration:.2f}s")
        
        # Processar falhas e erros
        for failure in failures:
            self.analyze_issue(app, 'failure', failure)
            
        for error in errors:
            self.analyze_issue(app, 'error', error)
    
    def analyze_issue(self, app, issue_type, issue):
        """Analisa uma falha ou erro e gera recomendações"""
        issue_info = {
            'app': app,
            'type': issue_type,
            'test': str(issue[0]) if len(issue) > 0 else 'Unknown',
            'message': str(issue[1]) if len(issue) > 1 else str(issue),
            'recommendation': self.get_recommendation(issue_type, str(issue))
        }
        
        self.results['issues_found'].append(issue_info)
    
    def get_recommendation(self, issue_type, message):
        """Gera recomendações baseadas no tipo de erro"""
        message_lower = message.lower()
        
        if 'import' in message_lower or 'module' in message_lower:
            return "Verificar imports e dependências do projeto"
        elif 'url' in message_lower or 'reverse' in message_lower:
            return "Verificar configuração de URLs e nomes de views"
        elif 'template' in message_lower:
            return "Verificar se os templates existem e estão no local correto"
        elif 'database' in message_lower or 'migration' in message_lower:
            return "Executar migrações do banco de dados"
        elif 'form' in message_lower:
            return "Verificar validação de formulários"
        elif 'authentication' in message_lower or 'login' in message_lower:
            return "Verificar configurações de autenticação"
        elif 'permission' in message_lower:
            return "Verificar permissões e decorators de views"
        else:
            return "Revisar o código e logs para identificar o problema"
    
    def run_integration_tests(self):
        """Executa testes de integração específicos"""
        print(f"\n🔗 Executando testes de integração...")
        print("-" * 40)
        
        try:
            # Testar se o servidor pode ser iniciado
            self.test_server_startup()
            
            # Testar configurações básicas
            self.test_basic_configuration()
            
            # Testar conectividade do banco
            self.test_database_connection()
            
        except Exception as e:
            print(f"❌ Erro nos testes de integração: {str(e)}")
            self.results['issues_found'].append({
                'type': 'integration_test_error',
                'message': str(e)
            })
    
    def test_server_startup(self):
        """Testa se o servidor Django pode ser iniciado"""
        try:
            from django.core.management import execute_from_command_line
            # Teste básico de configuração
            print("✅ Configuração do Django OK")
        except Exception as e:
            raise Exception(f"Erro na configuração do Django: {str(e)}")
    
    def test_basic_configuration(self):
        """Testa configurações básicas do projeto"""
        try:
            # Verificar se as configurações estão corretas
            assert hasattr(settings, 'SECRET_KEY'), "SECRET_KEY não configurada"
            assert hasattr(settings, 'DATABASES'), "Configuração de banco não encontrada"
            assert hasattr(settings, 'INSTALLED_APPS'), "INSTALLED_APPS não configurada"
            
            # Verificar se os apps estão instalados
            required_apps = ['core', 'accounts', 'models']
            for app in required_apps:
                assert app in settings.INSTALLED_APPS, f"App {app} não está em INSTALLED_APPS"
            
            print("✅ Configurações básicas OK")
            
        except Exception as e:
            raise Exception(f"Erro nas configurações básicas: {str(e)}")
    
    def test_database_connection(self):
        """Testa conexão com o banco de dados"""
        try:
            from django.db import connection
            with connection.cursor() as cursor:
                cursor.execute("SELECT 1")
            print("✅ Conexão com banco de dados OK")
            
        except Exception as e:
            raise Exception(f"Erro na conexão com banco: {str(e)}")
    
    def generate_report(self):
        """Gera relatório final dos testes"""
        print("\n" + "=" * 60)
        print("📊 RELATÓRIO FINAL DOS TESTES")
        print("=" * 60)
        
        # Estatísticas gerais
        total = self.results['total_tests']
        passed = self.results['passed']
        failed = self.results['failed']
        errors = self.results['errors']
        skipped = self.results['skipped']
        
        print(f"\n📈 ESTATÍSTICAS GERAIS:")
        print(f"   Total de testes: {total}")
        print(f"   ✅ Passaram: {passed}")
        print(f"   ❌ Falharam: {failed}")
        print(f"   ⚠️  Erros: {errors}")
        print(f"   ⏭️  Pulados: {skipped}")
        
        if total > 0:
            success_rate = (passed / total) * 100
            print(f"   📊 Taxa de sucesso: {success_rate:.1f}%")
        
        # Detalhes por app
        print(f"\n📋 DETALHES POR APP:")
        for app_result in self.results['test_details']:
            print(f"   {app_result['app']}: {app_result['passed']}/{app_result['total']} passaram "
                  f"({app_result['duration']}s)")
        
        # Problemas encontrados
        if self.results['issues_found']:
            print(f"\n🚨 PROBLEMAS ENCONTRADOS:")
            for i, issue in enumerate(self.results['issues_found'], 1):
                print(f"   {i}. {issue['type'].upper()}: {issue.get('message', 'N/A')}")
                if 'recommendation' in issue:
                    print(f"      💡 Recomendação: {issue['recommendation']}")
        
        # Recomendações gerais
        self.generate_general_recommendations()
        
        # Salvar relatório em arquivo
        self.save_report()
        
        # Resumo final
        print(f"\n🎯 RESUMO FINAL:")
        if failed == 0 and errors == 0:
            print("   🎉 Todos os testes passaram! O projeto está funcionando corretamente.")
        else:
            print(f"   ⚠️  {failed + errors} problemas encontrados. Revise as recomendações acima.")
        
        print(f"\n📄 Relatório salvo em: {BASE_DIR}/test_report.json")
    
    def generate_general_recommendations(self):
        """Gera recomendações gerais baseadas nos resultados"""
        print(f"\n💡 RECOMENDAÇÕES GERAIS:")
        
        if self.results['failed'] > 0 or self.results['errors'] > 0:
            print("   1. 🔧 Corrija os problemas identificados nos testes")
            print("   2. 🧪 Execute os testes novamente após as correções")
            print("   3. 📝 Documente as mudanças realizadas")
        
        if self.results['skipped'] > 0:
            print("   4. ⏭️  Revise os testes pulados e implemente se necessário")
        
        if self.results['total_tests'] < 50:  # Número arbitrário
            print("   5. 📈 Considere adicionar mais testes para melhor cobertura")
        
        print("   6. 🔄 Execute os testes regularmente durante o desenvolvimento")
        print("   7. 📊 Monitore a taxa de sucesso dos testes")
    
    def save_report(self):
        """Salva o relatório em arquivo JSON"""
        report_file = BASE_DIR / 'test_report.json'
        
        with open(report_file, 'w', encoding='utf-8') as f:
            json.dump(self.results, f, indent=2, ensure_ascii=False)
        
        # Também salvar em formato legível
        readable_report = BASE_DIR / 'test_report.txt'
        with open(readable_report, 'w', encoding='utf-8') as f:
            f.write("RELATÓRIO DE TESTES - PROJETO ALLURE\n")
            f.write("=" * 50 + "\n\n")
            f.write(f"Data/Hora: {self.results['timestamp']}\n\n")
            
            f.write("ESTATÍSTICAS GERAIS:\n")
            f.write(f"Total de testes: {self.results['total_tests']}\n")
            f.write(f"Passaram: {self.results['passed']}\n")
            f.write(f"Falharam: {self.results['failed']}\n")
            f.write(f"Erros: {self.results['errors']}\n")
            f.write(f"Pulados: {self.results['skipped']}\n\n")
            
            if self.results['issues_found']:
                f.write("PROBLEMAS ENCONTRADOS:\n")
                for i, issue in enumerate(self.results['issues_found'], 1):
                    f.write(f"{i}. {issue['type'].upper()}: {issue.get('message', 'N/A')}\n")
                    if 'recommendation' in issue:
                        f.write(f"   Recomendação: {issue['recommendation']}\n")
                f.write("\n")


def main():
    """Função principal"""
    print("🧪 SISTEMA DE TESTES - PROJETO ALLURE")
    print("=" * 60)
    
    # Verificar se estamos no diretório correto
    if not (BASE_DIR / 'manage.py').exists():
        print("❌ Erro: Execute este script no diretório raiz do projeto Django")
        sys.exit(1)
    
    # Criar e executar o test runner
    runner = TestRunner()
    
    try:
        runner.run_all_tests()
    except KeyboardInterrupt:
        print("\n⏹️  Execução interrompida pelo usuário")
        sys.exit(1)
    except Exception as e:
        print(f"\n💥 Erro crítico: {str(e)}")
        sys.exit(1)


if __name__ == '__main__':
    main() 