#!/usr/bin/env python
"""
Script principal para executar todos os testes do projeto Allure
Combina testes unitários, de funcionalidade e gera relatório consolidado
"""

import os
import sys
import subprocess
import json
import time
from datetime import datetime
from pathlib import Path

# Adicionar o diretório do projeto ao path
BASE_DIR = Path(__file__).resolve().parent
sys.path.insert(0, str(BASE_DIR))

# Configurações do Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')

import django
django.setup()


class AllureTestSuite:
    def __init__(self):
        self.results = {
            'timestamp': datetime.now().isoformat(),
            'test_suite': 'Allure Project Test Suite',
            'total_tests': 0,
            'passed': 0,
            'failed': 0,
            'errors': 0,
            'skipped': 0,
            'test_suites': {},
            'issues_found': [],
            'recommendations': [],
            'execution_time': 0
        }
    
    def run_complete_test_suite(self):
        """Executa toda a suíte de testes"""
        print("🧪 SUÍTE COMPLETA DE TESTES - PROJETO ALLURE")
        print("=" * 70)
        print("Este script executará todos os tipos de testes disponíveis")
        print("e gerará um relatório consolidado com recomendações.")
        print("=" * 70)
        
        start_time = time.time()
        
        # 1. Testes unitários do Django
        print("\n📋 1. EXECUTANDO TESTES UNITÁRIOS...")
        self.run_django_tests()
        
        # 2. Testes de funcionalidade
        print("\n🔍 2. EXECUTANDO TESTES DE FUNCIONALIDADE...")
        self.run_functionality_tests()
        
        # 3. Testes de integração
        print("\n🔗 3. EXECUTANDO TESTES DE INTEGRAÇÃO...")
        self.run_integration_tests()
        
        # 4. Testes de segurança
        print("\n🛡️ 4. EXECUTANDO TESTES DE SEGURANÇA...")
        self.run_security_tests()
        
        # 5. Verificações de qualidade
        print("\n📊 5. EXECUTANDO VERIFICAÇÕES DE QUALIDADE...")
        self.run_quality_checks()
        
        end_time = time.time()
        self.results['execution_time'] = round(end_time - start_time, 2)
        
        # Gerar relatório consolidado
        self.generate_consolidated_report()
    
    def run_django_tests(self):
        """Executa testes unitários do Django"""
        try:
            print("   Executando testes do app 'core'...")
            result = self.run_django_test_app('core')
            self.results['test_suites']['django_core'] = result
            
            print("   Executando testes do app 'accounts'...")
            result = self.run_django_test_app('accounts')
            self.results['test_suites']['django_accounts'] = result
            
            print("   Executando testes do app 'models'...")
            result = self.run_django_test_app('models')
            self.results['test_suites']['django_models'] = result
            
        except Exception as e:
            print(f"   ❌ Erro nos testes Django: {str(e)}")
            self.results['test_suites']['django_tests'] = {
                'status': 'error',
                'error': str(e)
            }
    
    def run_django_test_app(self, app_name):
        """Executa testes de um app específico"""
        try:
            from django.test.utils import get_runner
            from django.conf import settings
            
            TestRunner = get_runner(settings)
            test_runner = TestRunner()
            
            start_time = time.time()
            result = test_runner.run_tests([app_name])
            end_time = time.time()
            
            # Processar resultado
            failures = getattr(result, 'failures', [])
            errors = getattr(result, 'errors', [])
            skipped = getattr(result, 'skipped', [])
            total = getattr(result, 'testsRun', len(failures) + len(errors) + len(skipped))
            passed = total - len(failures) - len(errors) - len(skipped)
            
            return {
                'status': 'completed',
                'total': total,
                'passed': passed,
                'failed': len(failures),
                'errors': len(errors),
                'skipped': len(skipped),
                'duration': round(end_time - start_time, 2),
                'failures': [str(f) for f in failures],
                'errors': [str(e) for e in errors]
            }
            
        except Exception as e:
            return {
                'status': 'error',
                'error': str(e)
            }
    
    def run_functionality_tests(self):
        """Executa testes de funcionalidade"""
        try:
            # Importar e executar o testador de funcionalidade
            from test_functionality import FunctionalityTester
            
            tester = FunctionalityTester()
            tester.run_all_tests()
            
            # Capturar resultados
            self.results['test_suites']['functionality'] = {
                'status': 'completed',
                'stats': {
                    'total': tester.results['total_tests'],
                    'passed': tester.results['passed'],
                    'failed': tester.results['failed'],
                    'errors': tester.results['errors'],
                    'skipped': 0
                }
            }
            
        except Exception as e:
            print(f"   ❌ Erro nos testes de funcionalidade: {str(e)}")
            self.results['test_suites']['functionality'] = {
                'status': 'error',
                'error': str(e)
            }
    
    def run_integration_tests(self):
        """Executa testes de integração"""
        try:
            print("   Testando integração entre componentes...")
            
            integration_results = {
                'status': 'completed',
                'tests': [],
                'passed': 0,
                'failed': 0
            }
            
            # Teste 1: Verificar se o banco está funcionando
            try:
                from django.db import connection
                with connection.cursor() as cursor:
                    cursor.execute("SELECT 1")
                integration_results['tests'].append({
                    'name': 'Database Connection',
                    'status': 'passed'
                })
                integration_results['passed'] += 1
            except Exception as e:
                integration_results['tests'].append({
                    'name': 'Database Connection',
                    'status': 'failed',
                    'error': str(e)
                })
                integration_results['failed'] += 1
            
            # Teste 2: Verificar se as URLs estão configuradas
            try:
                from django.urls import reverse
                reverse('core:home')
                integration_results['tests'].append({
                    'name': 'URL Configuration',
                    'status': 'passed'
                })
                integration_results['passed'] += 1
            except Exception as e:
                integration_results['tests'].append({
                    'name': 'URL Configuration',
                    'status': 'failed',
                    'error': str(e)
                })
                integration_results['failed'] += 1
            
            # Teste 3: Verificar se os templates existem
            try:
                from django.conf import settings
                template_dirs = settings.TEMPLATES[0]['DIRS']
                base_template = Path(template_dirs[0]) / 'base.html'
                if base_template.exists():
                    integration_results['tests'].append({
                        'name': 'Template Files',
                        'status': 'passed'
                    })
                    integration_results['passed'] += 1
                else:
                    integration_results['tests'].append({
                        'name': 'Template Files',
                        'status': 'failed',
                        'error': 'base.html não encontrado'
                    })
                    integration_results['failed'] += 1
            except Exception as e:
                integration_results['tests'].append({
                    'name': 'Template Files',
                    'status': 'failed',
                    'error': str(e)
                })
                integration_results['failed'] += 1
            
            self.results['test_suites']['integration'] = integration_results
            
        except Exception as e:
            print(f"   ❌ Erro nos testes de integração: {str(e)}")
            self.results['test_suites']['integration'] = {
                'status': 'error',
                'error': str(e)
            }
    
    def run_security_tests(self):
        """Executa testes de segurança"""
        try:
            print("   Testando aspectos de segurança...")
            
            security_results = {
                'status': 'completed',
                'tests': [],
                'passed': 0,
                'failed': 0
            }
            
            # Teste 1: Verificar SECRET_KEY
            try:
                from django.conf import settings
                if settings.SECRET_KEY and settings.SECRET_KEY != 'django-insecure-7841@=4)h#i2(nfb9dbbrf6nzc1xayi@#)elu)ms=+11hg9-ca':
                    security_results['tests'].append({
                        'name': 'SECRET_KEY Configuration',
                        'status': 'passed'
                    })
                    security_results['passed'] += 1
                else:
                    security_results['tests'].append({
                        'name': 'SECRET_KEY Configuration',
                        'status': 'failed',
                        'error': 'SECRET_KEY padrão detectada'
                    })
                    security_results['failed'] += 1
            except Exception as e:
                security_results['tests'].append({
                    'name': 'SECRET_KEY Configuration',
                    'status': 'failed',
                    'error': str(e)
                })
                security_results['failed'] += 1
            
            # Teste 2: Verificar DEBUG
            try:
                from django.conf import settings
                if not settings.DEBUG:
                    security_results['tests'].append({
                        'name': 'DEBUG Mode',
                        'status': 'passed'
                    })
                    security_results['passed'] += 1
                else:
                    security_results['tests'].append({
                        'name': 'DEBUG Mode',
                        'status': 'warning',
                        'error': 'DEBUG está ativado (aceitável para desenvolvimento)'
                    })
                    security_results['passed'] += 1
            except Exception as e:
                security_results['tests'].append({
                    'name': 'DEBUG Mode',
                    'status': 'failed',
                    'error': str(e)
                })
                security_results['failed'] += 1
            
            # Teste 3: Verificar ALLOWED_HOSTS
            try:
                from django.conf import settings
                if settings.ALLOWED_HOSTS and '*' not in settings.ALLOWED_HOSTS:
                    security_results['tests'].append({
                        'name': 'ALLOWED_HOSTS Configuration',
                        'status': 'passed'
                    })
                    security_results['passed'] += 1
                else:
                    security_results['tests'].append({
                        'name': 'ALLOWED_HOSTS Configuration',
                        'status': 'warning',
                        'error': 'ALLOWED_HOSTS muito permissivo (aceitável para desenvolvimento)'
                    })
                    security_results['passed'] += 1
            except Exception as e:
                security_results['tests'].append({
                    'name': 'ALLOWED_HOSTS Configuration',
                    'status': 'failed',
                    'error': str(e)
                })
                security_results['failed'] += 1
            
            self.results['test_suites']['security'] = security_results
            
        except Exception as e:
            print(f"   ❌ Erro nos testes de segurança: {str(e)}")
            self.results['test_suites']['security'] = {
                'status': 'error',
                'error': str(e)
            }
    
    def run_quality_checks(self):
        """Executa verificações de qualidade de código"""
        try:
            print("   Executando verificações de qualidade...")
            
            quality_results = {
                'status': 'completed',
                'checks': [],
                'passed': 0,
                'failed': 0
            }
            
            # Verificação 1: Estrutura do projeto
            try:
                required_files = ['manage.py', 'allure_project/settings.py', 'allure_project/urls.py']
                missing_files = []
                
                for file_path in required_files:
                    if not (BASE_DIR / file_path).exists():
                        missing_files.append(file_path)
                
                if not missing_files:
                    quality_results['checks'].append({
                        'name': 'Project Structure',
                        'status': 'passed'
                    })
                    quality_results['passed'] += 1
                else:
                    quality_results['checks'].append({
                        'name': 'Project Structure',
                        'status': 'failed',
                        'error': f'Arquivos faltando: {", ".join(missing_files)}'
                    })
                    quality_results['failed'] += 1
            except Exception as e:
                quality_results['checks'].append({
                    'name': 'Project Structure',
                    'status': 'failed',
                    'error': str(e)
                })
                quality_results['failed'] += 1
            
            # Verificação 2: Apps instalados
            try:
                from django.conf import settings
                required_apps = ['core', 'accounts', 'models']
                missing_apps = []
                
                for app in required_apps:
                    if app not in settings.INSTALLED_APPS:
                        missing_apps.append(app)
                
                if not missing_apps:
                    quality_results['checks'].append({
                        'name': 'Installed Apps',
                        'status': 'passed'
                    })
                    quality_results['passed'] += 1
                else:
                    quality_results['checks'].append({
                        'name': 'Installed Apps',
                        'status': 'failed',
                        'error': f'Apps faltando: {", ".join(missing_apps)}'
                    })
                    quality_results['failed'] += 1
            except Exception as e:
                quality_results['checks'].append({
                    'name': 'Installed Apps',
                    'status': 'failed',
                    'error': str(e)
                })
                quality_results['failed'] += 1
            
            # Verificação 3: Migrações
            try:
                from django.core.management import call_command
                from io import StringIO
                
                out = StringIO()
                call_command('showmigrations', stdout=out)
                migrations_output = out.getvalue()
                
                if 'No migrations' not in migrations_output:
                    quality_results['checks'].append({
                        'name': 'Database Migrations',
                        'status': 'passed'
                    })
                    quality_results['passed'] += 1
                else:
                    quality_results['checks'].append({
                        'name': 'Database Migrations',
                        'status': 'warning',
                        'error': 'Nenhuma migração encontrada'
                    })
                    quality_results['passed'] += 1
            except Exception as e:
                quality_results['checks'].append({
                    'name': 'Database Migrations',
                    'status': 'failed',
                    'error': str(e)
                })
                quality_results['failed'] += 1
            
            self.results['test_suites']['quality'] = quality_results
            
        except Exception as e:
            print(f"   ❌ Erro nas verificações de qualidade: {str(e)}")
            self.results['test_suites']['quality'] = {
                'status': 'error',
                'error': str(e)
            }
    
    def generate_consolidated_report(self):
        """Gera relatório consolidado"""
        print("\n" + "=" * 70)
        print("📊 RELATÓRIO CONSOLIDADO - SUÍTE DE TESTES ALLURE")
        print("=" * 70)
        
        # Calcular estatísticas gerais
        self.calculate_overall_statistics()
        
        # Exibir estatísticas
        self.display_statistics()
        
        # Exibir detalhes por suíte
        self.display_suite_details()
        
        # Exibir problemas encontrados
        self.display_issues()
        
        # Exibir recomendações
        self.display_recommendations()
        
        # Salvar relatório
        self.save_consolidated_report()
        
        # Resumo final
        self.display_final_summary()
    
    def calculate_overall_statistics(self):
        """Calcula estatísticas gerais"""
        total_tests = 0
        total_passed = 0
        total_failed = 0
        total_errors = 0
        total_skipped = 0
        
        for suite_name, suite_result in self.results['test_suites'].items():
            if isinstance(suite_result, dict) and 'stats' in suite_result:
                stats = suite_result['stats']
                if isinstance(stats, dict):
                    total_tests += stats.get('total', 0)
                    total_passed += stats.get('passed', 0)
                    total_failed += stats.get('failed', 0)
                    total_errors += stats.get('errors', 0)
                    total_skipped += stats.get('skipped', 0)
        
        self.results['total_tests'] = total_tests
        self.results['passed'] = total_passed
        self.results['failed'] = total_failed
        self.results['errors'] = total_errors
        self.results['skipped'] = total_skipped
    
    def display_statistics(self):
        """Exibe estatísticas gerais"""
        total = self.results['total_tests']
        passed = self.results['passed']
        failed = self.results['failed']
        errors = self.results['errors']
        skipped = self.results['skipped']
        execution_time = self.results['execution_time']
        
        print(f"\n📈 ESTATÍSTICAS GERAIS:")
        print(f"   ⏱️  Tempo de execução: {execution_time}s")
        print(f"   📊 Total de testes: {total}")
        print(f"   ✅ Passaram: {passed}")
        print(f"   ❌ Falharam: {failed}")
        print(f"   ⚠️  Erros: {errors}")
        print(f"   ⏭️  Pulados: {skipped}")
        
        if total > 0:
            success_rate = (passed / total) * 100
            print(f"   🎯 Taxa de sucesso: {success_rate:.1f}%")
    
    def display_suite_details(self):
        """Exibe detalhes de cada suíte de testes"""
        print(f"\n📋 DETALHES POR SUÍTE DE TESTES:")
        
        for suite_name, suite_data in self.results['test_suites'].items():
            print(f"\n   🔹 {suite_name.upper()}:")
            
            if suite_data.get('status') == 'completed':
                total = suite_data.get('total', 0)
                passed = suite_data.get('passed', 0)
                failed = suite_data.get('failed', 0)
                duration = suite_data.get('duration', 0)
                
                print(f"      Status: ✅ Concluído")
                print(f"      Testes: {passed}/{total} passaram")
                print(f"      Duração: {duration}s")
                
                if failed > 0:
                    print(f"      ❌ {failed} falharam")
            else:
                print(f"      Status: ❌ Erro")
                print(f"      Erro: {suite_data.get('error', 'Erro desconhecido')}")
    
    def display_issues(self):
        """Exibe problemas encontrados"""
        issues_found = []
        
        # Coletar problemas de todas as suítes
        for suite_name, suite_data in self.results['test_suites'].items():
            if suite_data.get('status') == 'completed':
                # Problemas de falhas
                failures = suite_data.get('failures', [])
                for failure in failures:
                    issues_found.append({
                        'suite': suite_name,
                        'type': 'failure',
                        'message': failure
                    })
                
                # Problemas de erros
                errors = suite_data.get('errors', [])
                for error in errors:
                    issues_found.append({
                        'suite': suite_name,
                        'type': 'error',
                        'message': error
                    })
                
                # Problemas específicos de funcionalidade
                if suite_name == 'functionality':
                    issues = suite_data.get('issues', [])
                    for issue in issues:
                        issues_found.append({
                            'suite': suite_name,
                            'type': 'functionality',
                            'message': f"{issue['test']}: {issue['error']}"
                        })
        
        if issues_found:
            print(f"\n🚨 PROBLEMAS ENCONTRADOS ({len(issues_found)}):")
            for i, issue in enumerate(issues_found[:10], 1):  # Limitar a 10 problemas
                print(f"   {i}. [{issue['suite']}] {issue['type'].upper()}: {issue['message'][:100]}...")
            
            if len(issues_found) > 10:
                print(f"   ... e mais {len(issues_found) - 10} problemas")
        else:
            print(f"\n🎉 Nenhum problema encontrado!")
    
    def display_recommendations(self):
        """Exibe recomendações baseadas nos resultados"""
        print(f"\n💡 RECOMENDAÇÕES:")
        
        total = self.results['total_tests']
        passed = self.results['passed']
        failed = self.results['failed']
        
        if failed > 0:
            print("   1. 🔧 Corrija os problemas identificados nos testes")
            print("   2. 🧪 Execute os testes novamente após as correções")
            print("   3. 📝 Documente as mudanças realizadas")
        
        if total < 50:
            print("   4. 📈 Considere adicionar mais testes para melhor cobertura")
        
        # Verificar problemas de segurança
        security_suite = self.results['test_suites'].get('security', {})
        if security_suite.get('status') == 'completed':
            security_failed = security_suite.get('failed', 0)
            if security_failed > 0:
                print("   5. 🛡️  Revise as configurações de segurança")
        
        print("   6. 🔄 Execute esta suíte regularmente durante o desenvolvimento")
        print("   7. 📊 Monitore a qualidade do código continuamente")
    
    def display_final_summary(self):
        """Exibe resumo final"""
        print(f"\n🎯 RESUMO FINAL:")
        
        total = self.results['total_tests']
        passed = self.results['passed']
        failed = self.results['failed']
        errors = self.results['errors']
        
        if failed == 0 and errors == 0:
            print("   🎉 EXCELENTE! Todos os testes passaram!")
            print("   O projeto está funcionando corretamente.")
        elif failed + errors < 5:
            print("   ✅ BOM! A maioria dos testes passou.")
            print("   Alguns ajustes menores são necessários.")
        else:
            print("   ⚠️  ATENÇÃO! Vários problemas foram encontrados.")
            print("   Revise e corrija os problemas antes de prosseguir.")
        
        print(f"\n📄 Relatórios salvos em:")
        print(f"   - {BASE_DIR}/consolidated_test_report.json")
        print(f"   - {BASE_DIR}/consolidated_test_report.txt")
    
    def save_consolidated_report(self):
        """Salva o relatório consolidado"""
        # Salvar em JSON
        json_file = BASE_DIR / 'consolidated_test_report.json'
        with open(json_file, 'w', encoding='utf-8') as f:
            json.dump(self.results, f, indent=2, ensure_ascii=False)
        
        # Salvar em formato legível
        txt_file = BASE_DIR / 'consolidated_test_report.txt'
        with open(txt_file, 'w', encoding='utf-8') as f:
            f.write("RELATÓRIO CONSOLIDADO DE TESTES - PROJETO ALLURE\n")
            f.write("=" * 60 + "\n\n")
            f.write(f"Data/Hora: {self.results['timestamp']}\n")
            f.write(f"Tempo de execução: {self.results['execution_time']}s\n\n")
            
            f.write("ESTATÍSTICAS GERAIS:\n")
            f.write(f"Total de testes: {self.results['total_tests']}\n")
            f.write(f"Passaram: {self.results['passed']}\n")
            f.write(f"Falharam: {self.results['failed']}\n")
            f.write(f"Erros: {self.results['errors']}\n")
            f.write(f"Pulados: {self.results['skipped']}\n\n")
            
            f.write("DETALHES POR SUÍTE:\n")
            for suite_name, suite_data in self.results['test_suites'].items():
                f.write(f"\n{suite_name.upper()}:\n")
                if suite_data.get('status') == 'completed':
                    f.write(f"  Status: Concluído\n")
                    f.write(f"  Testes: {suite_data.get('passed', 0)}/{suite_data.get('total', 0)} passaram\n")
                else:
                    f.write(f"  Status: Erro\n")
                    f.write(f"  Erro: {suite_data.get('error', 'Erro desconhecido')}\n")


def main():
    """Função principal"""
    print("🧪 SUÍTE COMPLETA DE TESTES - PROJETO ALLURE")
    print("=" * 70)
    
    # Verificar se estamos no diretório correto
    if not (BASE_DIR / 'manage.py').exists():
        print("❌ Erro: Execute este script no diretório raiz do projeto Django")
        sys.exit(1)
    
    # Criar e executar a suíte de testes
    test_suite = AllureTestSuite()
    
    try:
        test_suite.run_complete_test_suite()
    except KeyboardInterrupt:
        print("\n⏹️  Execução interrompida pelo usuário")
        sys.exit(1)
    except Exception as e:
        print(f"\n💥 Erro crítico: {str(e)}")
        sys.exit(1)


if __name__ == '__main__':
    main() 