#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Script para processar o arquivo CSV de municípios da Receita Federal
e gerar o arquivo JavaScript com todos os 5.573 municípios brasileiros
"""

import csv
import os

def processar_csv_municipios():
    """Processa o CSV e gera o JavaScript com todos os municípios"""
    
    # Dicionário para armazenar municípios por estado
    municipios_por_estado = {}
    
    # Ler o arquivo CSV
    csv_path = 'municipios.csv'
    
    if not os.path.exists(csv_path):
        print(f"Erro: Arquivo {csv_path} não encontrado!")
        return
    
    print("Processando arquivo CSV de municípios...")
    
    # Tentar diferentes encodings
    encodings = ['latin1', 'iso-8859-1', 'cp1252', 'utf-8']
    
    for encoding in encodings:
        try:
            with open(csv_path, 'r', encoding=encoding) as arquivo:
                # Pular o cabeçalho
                next(arquivo)
                
                # Ler cada linha
                for linha in arquivo:
                    # Dividir por ponto e vírgula
                    campos = linha.strip().split(';')
                    
                    if len(campos) >= 5:
                        # Extrair informações
                        codigo_tom = campos[0]
                        codigo_ibge = campos[1]
                        municipio_tom = campos[2]
                        municipio_ibge = campos[3]
                        uf = campos[4]
                        
                        # Usar o nome do município do IBGE (mais limpo)
                        nome_municipio = municipio_ibge.strip()
                        
                        # Adicionar ao dicionário
                        if uf not in municipios_por_estado:
                            municipios_por_estado[uf] = []
                        
                        municipios_por_estado[uf].append(nome_municipio)
            
            print(f"Arquivo processado com sucesso usando encoding: {encoding}")
            break
            
        except UnicodeDecodeError:
            print(f"Encoding {encoding} falhou, tentando próximo...")
            continue
        except Exception as e:
            print(f"Erro com encoding {encoding}: {e}")
            continue
    else:
        print("Erro: Não foi possível processar o arquivo com nenhum encoding conhecido!")
        return
    
    print(f"Total de estados encontrados: {len(municipios_por_estado)}")
    
    # Gerar o arquivo JavaScript
    gerar_arquivo_js(municipios_por_estado)

def gerar_arquivo_js(municipios_por_estado):
    """Gera o arquivo JavaScript com todos os municípios"""
    
    js_content = '''// Sistema completo de municípios brasileiros
// Baseado no arquivo oficial da Receita Federal
// Total: 5.573 municípios

// Estados brasileiros com siglas
const ESTADOS_BRASIL = [
    { sigla: 'AC', nome: 'Acre' },
    { sigla: 'AL', nome: 'Alagoas' },
    { sigla: 'AP', nome: 'Amapá' },
    { sigla: 'AM', nome: 'Amazonas' },
    { sigla: 'BA', nome: 'Bahia' },
    { sigla: 'CE', nome: 'Ceará' },
    { sigla: 'DF', nome: 'Distrito Federal' },
    { sigla: 'ES', nome: 'Espírito Santo' },
    { sigla: 'GO', nome: 'Goiás' },
    { sigla: 'MA', nome: 'Maranhão' },
    { sigla: 'MT', nome: 'Mato Grosso' },
    { sigla: 'MS', nome: 'Mato Grosso do Sul' },
    { sigla: 'MG', nome: 'Minas Gerais' },
    { sigla: 'PA', nome: 'Pará' },
    { sigla: 'PB', nome: 'Paraíba' },
    { sigla: 'PR', nome: 'Paraná' },
    { sigla: 'PE', nome: 'Pernambuco' },
    { sigla: 'PI', nome: 'Piauí' },
    { sigla: 'RJ', nome: 'Rio de Janeiro' },
    { sigla: 'RN', nome: 'Rio Grande do Norte' },
    { sigla: 'RS', nome: 'Rio Grande do Sul' },
    { sigla: 'RO', nome: 'Rondônia' },
    { sigla: 'RR', nome: 'Roraima' },
    { sigla: 'SC', nome: 'Santa Catarina' },
    { sigla: 'SP', nome: 'São Paulo' },
    { sigla: 'SE', nome: 'Sergipe' },
    { sigla: 'TO', nome: 'Tocantins' }
];

// Dados completos dos municípios (estrutura otimizada)
const MUNICIPIOS_COMPLETOS = {
'''
    
    # Adicionar municípios por estado
    for uf, municipios in sorted(municipios_por_estado.items()):
        js_content += f"    '{uf}': [\n"
        
        # Ordenar municípios alfabeticamente
        municipios_ordenados = sorted(municipios)
        
        for i, municipio in enumerate(municipios_ordenados):
            # Escapar aspas simples no nome do município
            municipio_escaped = municipio.replace("'", "\\'")
            
            if i == len(municipios_ordenados) - 1:
                js_content += f"        '{municipio_escaped}'\n"
            else:
                js_content += f"        '{municipio_escaped}',\n"
        
        js_content += "    ],\n"
    
    # Adicionar o resto do código JavaScript
    js_content += '''};

// Função para popular o select de estados
function popularEstados() {
    console.log('Populando estados...');
    const selectEstado = document.getElementById('estado');
    if (!selectEstado) {
        console.log('Select de estado não encontrado');
        return;
    }
    
    console.log('Select de estado encontrado, populando...');
    
    selectEstado.innerHTML = '<option value="">Selecione o estado</option>';
    
    ESTADOS_BRASIL.forEach(estado => {
        const option = document.createElement('option');
        option.value = estado.sigla;
        option.textContent = `${estado.sigla} - ${estado.nome}`;
        selectEstado.appendChild(option);
    });
    
    console.log('Estados populados:', ESTADOS_BRASIL.length);
}

// Função para filtrar municípios por estado
function filtrarMunicipios(estado) {
    console.log('Filtrando municípios para estado:', estado);
    if (!estado) {
        console.log('Estado não fornecido');
        return [];
    }
    
    const municipios = MUNICIPIOS_COMPLETOS[estado] || [];
    console.log('Municípios encontrados para', estado, ':', municipios.length);
    return municipios.sort();
}

// Função para criar autocompletar de cidade
function criarAutocompletarCidade() {
    console.log('Criando autocompletar de cidade...');
    const campoCidade = document.getElementById('cidade');
    const campoEstado = document.getElementById('estado');
    
    if (!campoCidade || !campoEstado) {
        console.log('Campos cidade ou estado não encontrados');
        return;
    }
    
    console.log('Campos encontrados, criando autocompletar...');
    
    // Criar container para sugestões
    const sugestoesContainer = document.createElement('div');
    sugestoesContainer.id = 'sugestoes-cidade';
    sugestoesContainer.className = 'sugestoes-autocompletar';
    campoCidade.parentNode.appendChild(sugestoesContainer);
    
    // Evento de digitação na cidade
    campoCidade.addEventListener('input', function() {
        const valor = this.value.toLowerCase();
        const estado = campoEstado.value;
        
        console.log('Digitação na cidade:', valor, 'Estado:', estado);
        
        if (valor.length < 2 || !estado) {
            sugestoesContainer.innerHTML = '';
            sugestoesContainer.style.display = 'none';
            return;
        }
        
        const municipios = filtrarMunicipios(estado);
        console.log('Municípios encontrados para', estado, ':', municipios.length);
        
        const sugestoes = municipios.filter(municipio => 
            municipio.toLowerCase().includes(valor)
        ).slice(0, 10);
        
        console.log('Sugestões filtradas:', sugestoes);
        
        if (sugestoes.length > 0) {
            sugestoesContainer.innerHTML = sugestoes.map(municipio => 
                `<div class="sugestao-item" onclick="selecionarCidade('${municipio}')">${municipio}</div>`
            ).join('');
            sugestoesContainer.style.display = 'block';
        } else {
            sugestoesContainer.innerHTML = '';
            sugestoesContainer.style.display = 'none';
        }
    });
    
    // Evento de mudança no estado
    campoEstado.addEventListener('change', function() {
        console.log('Estado mudou para:', this.value);
        campoCidade.value = '';
        sugestoesContainer.innerHTML = '';
        sugestoesContainer.style.display = 'none';
        
        // Limpar validação da cidade
        campoCidade.classList.remove('campo-valido', 'campo-invalido');
        const feedback = document.getElementById('feedback-cidade');
        if (feedback) feedback.innerHTML = '';
    });
    
    // Esconder sugestões ao clicar fora
    document.addEventListener('click', function(e) {
        if (!campoCidade.contains(e.target) && !sugestoesContainer.contains(e.target)) {
            sugestoesContainer.style.display = 'none';
        }
    });
    
    console.log('Autocompletar de cidade criado com sucesso');
}

// Função para selecionar cidade
function selecionarCidade(cidade) {
    const campoCidade = document.getElementById('cidade');
    const sugestoesContainer = document.getElementById('sugestoes-cidade');
    
    if (campoCidade) {
        campoCidade.value = cidade;
        
        // Disparar evento de validação
        const event = new Event('input');
        campoCidade.dispatchEvent(event);
    }
    
    if (sugestoesContainer) {
        sugestoesContainer.style.display = 'none';
    }
}

// Função para validar cidade
function validarCidade(cidade, estado) {
    console.log('Validando cidade:', cidade, 'Estado:', estado);
    if (!estado || !cidade) {
        console.log('Estado ou cidade vazios');
        return false;
    }
    
    const municipios = filtrarMunicipios(estado);
    const cidadeExiste = municipios.some(municipio => 
        municipio.toLowerCase() === cidade.toLowerCase()
    );
    
    console.log('Municípios disponíveis:', municipios.length);
    console.log('Cidade existe:', cidadeExiste);
    
    return cidadeExiste;
}

// Inicializar quando o DOM estiver carregado
document.addEventListener('DOMContentLoaded', function() {
    console.log('=== INICIALIZANDO SISTEMA COMPLETO DE MUNICÍPIOS ===');
    
    // Primeiro popular estados
    popularEstados();
    
    // Depois criar autocompletar de cidade
    setTimeout(() => {
        criarAutocompletarCidade();
    }, 100);
    
    console.log('=== SISTEMA COMPLETO DE MUNICÍPIOS INICIALIZADO ===');
});
'''
    
    # Salvar o arquivo JavaScript
    output_path = 'static/js/municipios_completo.js'
    
    # Criar diretório se não existir
    os.makedirs(os.path.dirname(output_path), exist_ok=True)
    
    with open(output_path, 'w', encoding='utf-8') as arquivo:
        arquivo.write(js_content)
    
    print(f"Arquivo JavaScript gerado: {output_path}")
    
    # Estatísticas
    total_municipios = sum(len(municipios) for municipios in municipios_por_estado.values())
    print(f"Total de municípios processados: {total_municipios}")
    
    for uf, municipios in sorted(municipios_por_estado.items()):
        print(f"{uf}: {len(municipios)} municípios")

if __name__ == "__main__":
    processar_csv_municipios()
