import mercadopago
from django.conf import settings
from django.utils import timezone
from datetime import timedelta
import logging
from .models import TransacaoPagamento, WebhookLog, Voucher, VoucherUso

logger = logging.getLogger(__name__)


class MercadoPagoService:
    """Serviço para integração com Mercado Pago"""
    
    def __init__(self):
        # Configurar SDK do Mercado Pago
        self.sdk = mercadopago.SDK(settings.MERCADO_PAGO_ACCESS_TOKEN)
        self.sandbox = getattr(settings, 'MERCADO_PAGO_SANDBOX', True)
    
    def criar_preferencia_pagamento(self, transacao):
        """Cria uma preferência de pagamento no Mercado Pago"""
        try:
            # Verificar se está em modo de teste
            if getattr(settings, 'MODO_TESTE_PAGAMENTO', False):
                return self._criar_preferencia_teste(transacao)
            
            # Preparar dados da preferência
            preference_data = {
                "items": [
                    {
                        "title": transacao.descricao or f"Pagamento {transacao.tipo}",
                        "quantity": 1,
                        "unit_price": float(transacao.valor),
                        "currency_id": "BRL"
                    }
                ],
                "external_reference": str(transacao.external_reference),
                "notification_url": f"{settings.SITE_URL}/payments/webhook/",
                "back_urls": {
                    "success": f"{settings.SITE_URL}/payments/success/{transacao.id}/",
                    "failure": f"{settings.SITE_URL}/payments/failure/{transacao.id}/",
                    "pending": f"{settings.SITE_URL}/payments/pending/{transacao.id}/"
                },
                "auto_return": "approved",
                "expires": True,
                "expiration_date_to": (timezone.now() + timedelta(hours=24)).isoformat(),
                "payment_methods": {
                    "excluded_payment_types": [
                        {"id": "ticket"}  # Excluir boleto se necessário
                    ],
                    "installments": 12
                }
            }
            
            # Adicionar dados do pagador se disponível
            if transacao.email_pagador:
                preference_data["payer"] = {
                    "email": transacao.email_pagador,
                    "name": transacao.nome_pagador or ""
                }
            
            # Criar preferência
            preference_response = self.sdk.preference().create(preference_data)
            
            if preference_response["status"] == 201:
                preference = preference_response["response"]
                transacao.preference_id = preference["id"]
                transacao.save()
                
                logger.info(f"Preferência criada com sucesso: {preference['id']}")
                return preference
            else:
                logger.error(f"Erro ao criar preferência: {preference_response}")
                return None
                
        except Exception as e:
            logger.error(f"Erro ao criar preferência de pagamento: {str(e)}")
            return None
    
    def _criar_preferencia_teste(self, transacao):
        """Cria uma preferência de teste (simulada)"""
        try:
            # Simular preferência do Mercado Pago
            preference_id = f"test_pref_{transacao.id}"
            transacao.preference_id = preference_id
            transacao.save()
            
            # Retornar dados simulados
            preference = {
                "id": preference_id,
                "init_point": f"{settings.SITE_URL}/payments/teste-pagamento/{transacao.id}/",
                "sandbox_init_point": f"{settings.SITE_URL}/payments/teste-pagamento/{transacao.id}/"
            }
            
            logger.info(f"Preferência de teste criada: {preference_id}")
            return preference
            
        except Exception as e:
            logger.error(f"Erro ao criar preferência de teste: {str(e)}")
            return None
    
    def processar_webhook(self, data):
        """Processa webhook recebido do Mercado Pago"""
        try:
            # Log do webhook
            webhook_log = WebhookLog.objects.create(
                tipo=data.get('type', 'other'),
                dados_recebidos=data
            )
            
            # Processar baseado no tipo
            if data.get('type') == 'payment':
                return self._processar_pagamento_webhook(data, webhook_log)
            elif data.get('type') == 'preference':
                return self._processar_preferencia_webhook(data, webhook_log)
            else:
                webhook_log.processado = True
                webhook_log.save()
                return True
                
        except Exception as e:
            logger.error(f"Erro ao processar webhook: {str(e)}")
            if 'webhook_log' in locals():
                webhook_log.erro = str(e)
                webhook_log.processado = True
                webhook_log.save()
            return False
    
    def _processar_pagamento_webhook(self, data, webhook_log):
        """Processa webhook de pagamento"""
        try:
            payment_id = data.get('data', {}).get('id')
            if not payment_id:
                webhook_log.erro = "Payment ID não encontrado"
                webhook_log.processado = True
                webhook_log.save()
                return False
            
            # Buscar informações do pagamento
            payment_response = self.sdk.payment().get(payment_id)
            
            if payment_response["status"] != 200:
                webhook_log.erro = f"Erro ao buscar pagamento: {payment_response}"
                webhook_log.processado = True
                webhook_log.save()
                return False
            
            payment_data = payment_response["response"]
            external_reference = payment_data.get('external_reference')
            
            # Buscar pagamento no banco
            try:
                transacao = TransacaoPagamento.objects.get(external_reference=external_reference)
            except TransacaoPagamento.DoesNotExist:
                webhook_log.erro = f"Transação não encontrada: {external_reference}"
                webhook_log.processado = True
                webhook_log.save()
                return False
            
            # Atualizar status do pagamento
            status_mp = payment_data.get('status')
            status_mapping = {
                'approved': 'aprovado',
                'pending': 'pendente',
                'rejected': 'rejeitado',
                'cancelled': 'cancelado',
                'in_process': 'processando',
                'refunded': 'cancelado'
            }
            
            novo_status = status_mapping.get(status_mp, 'pendente')
            transacao.status = novo_status
            transacao.payment_id = str(payment_id)
            
            if novo_status == 'aprovado':
                transacao.data_aprovacao = timezone.now()
                self._ativar_servico(transacao)
            
            transacao.save()
            
            # Marcar webhook como processado
            webhook_log.processado = True
            webhook_log.data_processamento = timezone.now()
            webhook_log.save()
            
            logger.info(f"Webhook processado com sucesso: {payment_id}")
            return True
            
        except Exception as e:
            logger.error(f"Erro ao processar webhook de pagamento: {str(e)}")
            webhook_log.erro = str(e)
            webhook_log.processado = True
            webhook_log.save()
            return False
    
    def _processar_preferencia_webhook(self, data, webhook_log):
        """Processa webhook de preferência"""
        webhook_log.processado = True
        webhook_log.data_processamento = timezone.now()
        webhook_log.save()
        return True
    
    def _ativar_servico(self, transacao):
        """Ativa o serviço após pagamento aprovado"""
        try:
            if transacao.tipo == 'plano_anuncio':
                self._ativar_plano_anuncio(transacao)
            
            logger.info(f"Serviço ativado: {transacao.tipo} - {transacao.id}")
            
        except Exception as e:
            logger.error(f"Erro ao ativar serviço: {str(e)}")
    
    def _ativar_plano_anuncio(self, transacao):
        """Ativa plano de anúncio"""
        try:
            PlanoAnuncioService.ativar_plano_anuncio(transacao)
        except Exception as e:
            logger.error(f"Erro ao ativar plano de anúncio: {str(e)}")
    
    def verificar_status_pagamento(self, payment_id):
        """Verifica status de um pagamento específico"""
        try:
            payment_response = self.sdk.payment().get(payment_id)
            
            if payment_response["status"] == 200:
                return payment_response["response"]
            else:
                logger.error(f"Erro ao verificar pagamento: {payment_response}")
                return None
                
        except Exception as e:
            logger.error(f"Erro ao verificar status do pagamento: {str(e)}")
            return None
    
    def cancelar_pagamento(self, transacao):
        """Cancela um pagamento pendente"""
        try:
            if transacao.payment_id and transacao.status == 'pendente':
                # Para pagamentos pendentes, podemos cancelar via API
                cancel_response = self.sdk.payment().cancel(transacao.payment_id)
                
                if cancel_response["status"] == 200:
                    transacao.status = 'cancelado'
                    transacao.save()
                    return True
                else:
                    logger.error(f"Erro ao cancelar pagamento: {cancel_response}")
                    return False
            else:
                # Para outros casos, apenas marcar como cancelado
                transacao.status = 'cancelado'
                transacao.save()
                return True
                
        except Exception as e:
            logger.error(f"Erro ao cancelar pagamento: {str(e)}")
            return False


class PlanoAnuncioService:
    """Serviço para gerenciar planos de anúncio"""
    
    @staticmethod
    def criar_pagamento_plano(modelo, plano_id, email_pagador=None, nome_pagador=None):
        """Cria um pagamento para um plano de anúncio"""
        try:
            from models.models import PlanoAnuncio
            
            plano = PlanoAnuncio.objects.get(id=plano_id, ativo=True)
            
            # Criar transação
            transacao = TransacaoPagamento.objects.create(
                tipo='plano_anuncio',
                valor=plano.preco,
                descricao=f'Plano {plano.nome} - {plano.duracao_dias} dias',
                modelo=modelo,
                usuario=modelo.user if modelo else None,
                email_pagador=email_pagador or (modelo.user.email if modelo and modelo.user else ''),
                nome_pagador=nome_pagador or (f"{modelo.user.first_name} {modelo.user.last_name}".strip() if modelo and modelo.user else '')
            )
            
            # Criar plano específico
            from .models import PlanoAnuncioPagamento
            plano_pagamento = PlanoAnuncioPagamento.objects.create(
                pagamento=transacao,
                plano=plano
            )
            
            return transacao
            
        except Exception as e:
            logger.error(f"Erro ao criar pagamento de plano: {str(e)}")
            return None
    
    @staticmethod
    def ativar_plano_anuncio(transacao):
        """Ativa o plano de anúncio após pagamento aprovado"""
        try:
            from .models import PlanoAnuncioPagamento
            
            plano_pagamento = PlanoAnuncioPagamento.objects.get(pagamento=transacao)
            modelo = transacao.modelo
            
            if not modelo:
                return False
            
            # Calcular datas
            data_inicio = timezone.now()
            data_fim = data_inicio + timedelta(days=plano_pagamento.plano.duracao_dias)
            
            # Atualizar plano
            plano_pagamento.data_inicio = data_inicio
            plano_pagamento.data_fim = data_fim
            plano_pagamento.save()
            
            # Atualizar modelo
            modelo.status = 'ativo'
            modelo.pagamento_aprovado = True
            modelo.data_pagamento = data_inicio
            modelo.proximo_pagamento = data_fim
            modelo.plano_ativo = plano_pagamento.plano
            modelo.plano_data_inicio = data_inicio
            modelo.plano_data_fim = data_fim
            
            # Adicionar plano ao ManyToManyField para exibição no admin
            modelo.planos_anuncio.add(plano_pagamento.plano)
            
            modelo.save()
            
            # Aplicar benefícios do plano
            PlanoAnuncioService._aplicar_beneficios_plano(modelo, plano_pagamento.plano)
            
            # Criar log do plano ativo
            PlanoAnuncioService._criar_log_plano_ativo(
                modelo=modelo,
                plano=plano_pagamento.plano,
                transacao=transacao,
                data_inicio=data_inicio,
                data_fim=data_fim,
                tipo_ativacao='pagamento'
            )
            
            return True
            
        except Exception as e:
            logger.error(f"Erro ao ativar plano de anúncio: {str(e)}")
            return False
    
    @staticmethod
    def _aplicar_beneficios_plano(modelo, plano):
        """Aplica os benefícios do plano ao modelo"""
        try:
            # Aplicar destaque se disponível (agora via plano_ativo)
            # Benefícios do plano são aplicados através do plano_ativo
            
            # Aplicar boost se disponível (agora através do plano ativo)
            if plano.boost_visualizacoes:
                modelo.plano_ativo = plano
                modelo.plano_data_inicio = timezone.now()
                modelo.plano_data_fim = timezone.now() + timedelta(days=plano.duracao_dias)
            
            modelo.save()
            
        except Exception as e:
            logger.error(f"Erro ao aplicar benefícios do plano: {str(e)}")
    
    @staticmethod
    def _criar_log_plano_ativo(modelo, plano, transacao, data_inicio, data_fim, tipo_ativacao):
        """Cria um log de plano ativo"""
        try:
            from models.models import PlanoAtivoLog
            
            PlanoAtivoLog.objects.create(
                modelo=modelo,
                plano=plano,
                tipo_ativacao=tipo_ativacao,
                data_inicio=data_inicio,
                data_fim=data_fim,
                valor_pago=transacao.valor if transacao else None,
                transacao_pagamento=transacao,
                observacoes=f'Plano ativado via {tipo_ativacao}'
            )
            
            logger.info(f"Log de plano criado: {modelo.nome_exibicao} - {plano.nome}")
            
        except Exception as e:
            logger.error(f"Erro ao criar log de plano: {str(e)}")


class VoucherService:
    """Serviço para gerenciar vouchers"""
    
    @staticmethod
    def validar_voucher(codigo, modelo):
        """Valida se um voucher pode ser usado por uma modelo"""
        try:
            voucher = Voucher.objects.get(codigo=codigo)
            
            if not voucher.pode_ser_usado_por(modelo):
                return None, "Voucher não pode ser usado por esta modelo"
            
            return voucher, None
            
        except Voucher.DoesNotExist:
            return None, "Voucher não encontrado"
        except Exception as e:
            logger.error(f"Erro ao validar voucher: {str(e)}")
            return None, "Erro interno ao validar voucher"
    
    @staticmethod
    def aplicar_voucher(voucher, modelo, valor_original):
        """Aplica um voucher e retorna o valor final"""
        try:
            valor_final = voucher.calcular_valor_final(valor_original)
            valor_desconto = valor_original - valor_final
            
            return {
                'valor_original': valor_original,
                'valor_desconto': valor_desconto,
                'valor_final': valor_final,
                'voucher': voucher
            }
            
        except Exception as e:
            logger.error(f"Erro ao aplicar voucher: {str(e)}")
            return None
    
    @staticmethod
    def registrar_uso_voucher(voucher, modelo, transacao, valor_original, valor_desconto, valor_final, observacoes=''):
        """Registra o uso de um voucher"""
        try:
            from .models import VoucherUso
            
            # Criar registro de uso
            uso = VoucherUso.objects.create(
                voucher=voucher,
                modelo=modelo,
                transacao=transacao,
                valor_original=valor_original,
                valor_desconto=valor_desconto,
                valor_final=valor_final,
                observacoes=observacoes
            )
            
            # Atualizar contador de uso do voucher
            voucher.quantidade_usada += 1
            voucher.save()
            
            logger.info(f"Uso de voucher registrado: {voucher.codigo} - {modelo.nome_exibicao}")
            return uso
            
        except Exception as e:
            logger.error(f"Erro ao registrar uso de voucher: {str(e)}")
            return None
    
    @staticmethod
    def gerar_codigo_voucher(prefixo='ALLURE', tamanho=8):
        """Gera um código único para voucher"""
        import random
        import string
        
        while True:
            sufixo = ''.join(random.choices(string.ascii_uppercase + string.digits, k=tamanho))
            codigo = f"{prefixo}{sufixo}"
            
            if not Voucher.objects.filter(codigo=codigo).exists():
                return codigo


# Instância global do serviço
mercado_pago_service = MercadoPagoService()
voucher_service = VoucherService() 