from django.test import TestCase
from django.contrib.auth.models import User
from django.core.files.uploadedfile import SimpleUploadedFile
from django.db import IntegrityError
from .models import Modelo, FotoModelo, Banner


class ModeloModelTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            altura=170,
            peso=60,
            cidade='São Paulo',
            estado='SP',
            sobre_mim='Descrição de teste',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )

    def test_modelo_creation(self):
        """Testa criação de modelo"""
        self.assertEqual(self.modelo.nome_exibicao, 'Modelo Teste')
        self.assertEqual(self.modelo.sobre_mim, 'Descrição de teste')
        self.assertEqual(self.modelo.status, 'pendente')
        self.assertEqual(self.modelo.user, self.user)

    def test_modelo_deactivation(self):
        """Testa desativação de modelo"""
        self.modelo.status = 'suspenso'
        self.modelo.save()
        self.assertEqual(self.modelo.status, 'suspenso')

    def test_modelo_ordering(self):
        """Testa ordenação dos modelos"""
        # Criar modelos com datas diferentes
        user2 = User.objects.create_user(username='testuser2', password='testpass123')
        user3 = User.objects.create_user(username='testuser3', password='testpass123')
        
        modelo2 = Modelo.objects.create(
            user=user2,
            nome_exibicao='Modelo 2',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        modelo3 = Modelo.objects.create(
            user=user3,
            nome_exibicao='Modelo 3',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        # Verificar ordenação por data de cadastro (mais recente primeiro)
        modelos = list(Modelo.objects.all())
        self.assertGreaterEqual(modelos[0].data_cadastro, modelos[1].data_cadastro)
        self.assertGreaterEqual(modelos[1].data_cadastro, modelos[2].data_cadastro)

    def test_modelo_user_relationship(self):
        """Testa relacionamento com usuário"""
        self.assertEqual(self.modelo.user, self.user)
        self.assertEqual(self.user.modelo_profile, self.modelo)

    def test_modelo_str_representation(self):
        """Testa representação string do modelo"""
        self.assertEqual(str(self.modelo), 'Modelo Teste')

    def test_modelo_choices_validation(self):
        """Testa validação das choices"""
        # Testar categoria válida
        self.modelo.categoria = 'FAN'
        self.modelo.save()
        self.assertEqual(self.modelo.categoria, 'FAN')
        
        # Testar etnia válida
        self.modelo.etnia = 'parda'
        self.modelo.save()
        self.assertEqual(self.modelo.etnia, 'parda')


class FotoModeloModelTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        self.image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        self.foto = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='galeria'
        )

    def test_foto_creation(self):
        """Testa criação de foto"""
        self.assertEqual(self.foto.categoria, 'galeria')
        self.assertEqual(self.foto.modelo, self.modelo)

    def test_foto_str_representation(self):
        """Testa representação string da foto"""
        self.assertIn('Modelo Teste', str(self.foto))

    def test_foto_categoria_choices(self):
        """Testa escolhas válidas para categoria de foto"""
        # Testar categoria válida
        foto_valida = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='seios'
        )
        self.assertEqual(foto_valida.categoria, 'seios')
        
        # Testar categoria inválida (deve usar valor padrão)
        foto_invalida = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='categoria_invalida'
        )
        # Como não há validação no modelo, o valor inválido será salvo
        self.assertEqual(foto_invalida.categoria, 'categoria_invalida')

    def test_foto_modelo_relationship(self):
        """Testa relacionamento entre foto e modelo"""
        self.assertEqual(self.foto.modelo, self.modelo)
        self.assertIn(self.foto, self.modelo.fotos.all())

    def test_foto_ordering(self):
        """Testa ordenação das fotos"""
        # Criar fotos com ordens diferentes
        foto2 = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='galeria',
            ordem=2
        )
        foto3 = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='galeria',
            ordem=1
        )
        
        # Verificar ordenação por ordem e data de upload
        fotos = list(FotoModelo.objects.all())
        self.assertEqual(fotos[0].ordem, 0)  # Primeira foto criada
        self.assertEqual(fotos[1].ordem, 1)  # foto3
        self.assertEqual(fotos[2].ordem, 2)  # foto2


class BannerModelTestCase(TestCase):
    def setUp(self):
        self.image = SimpleUploadedFile(
            "banner.jpg",
            b"fake-banner-content",
            content_type="image/jpeg"
        )
        
        self.banner = Banner.objects.create(
            titulo='Banner Teste',
            imagem=self.image,
            link='https://example.com',
            ordem=1
        )

    def test_banner_creation(self):
        """Testa criação de banner"""
        self.assertEqual(self.banner.titulo, 'Banner Teste')
        self.assertEqual(self.banner.link, 'https://example.com')
        self.assertEqual(self.banner.ordem, 1)
        self.assertTrue(self.banner.ativo)  # Ativo por padrão

    def test_banner_deactivation(self):
        """Testa desativação de banner"""
        self.banner.ativo = False
        self.banner.save()
        self.assertFalse(self.banner.ativo)

    def test_banner_ordering(self):
        """Testa ordenação dos banners"""
        banner2 = Banner.objects.create(
            titulo='Banner 2',
            imagem=self.image,
            ordem=0
        )
        
        banners = list(Banner.objects.all())
        self.assertEqual(banners[0].ordem, 0)  # banner2
        self.assertEqual(banners[1].ordem, 1)  # self.banner

    def test_banner_str_representation(self):
        """Testa representação string do banner"""
        self.assertEqual(str(self.banner), 'Banner Teste')


class ModelRelationshipsTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        self.image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        self.foto = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='galeria'
        )

    def test_modelo_foto_relationship(self):
        """Testa relacionamento entre modelo e foto"""
        self.assertEqual(self.foto.modelo, self.modelo)
        self.assertIn(self.foto, self.modelo.fotos.all())

    def test_cascade_delete_user(self):
        """Testa exclusão em cascata quando usuário é deletado"""
        self.user.delete()
        self.assertFalse(FotoModelo.objects.filter(id=self.foto.id).exists())

    def test_cascade_delete_modelo(self):
        """Testa exclusão em cascata quando modelo é deletado"""
        self.modelo.delete()
        self.assertFalse(FotoModelo.objects.filter(id=self.foto.id).exists())


class ModelValidationTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_foto_required_fields(self):
        """Testa campos obrigatórios da foto"""
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        # Tentar criar foto sem campos obrigatórios
        with self.assertRaises(IntegrityError):
            FotoModelo.objects.create(
                modelo=modelo,
                # Faltando campos obrigatórios
            )

    def test_banner_required_fields(self):
        """Testa campos obrigatórios do banner"""
        # Tentar criar banner sem campos obrigatórios
        with self.assertRaises(IntegrityError):
            Banner.objects.create(
                # Faltando campos obrigatórios
            )

    def test_modelo_required_fields(self):
        """Testa campos obrigatórios do modelo"""
        # Tentar criar modelo sem campos obrigatórios
        with self.assertRaises(IntegrityError):
            Modelo.objects.create(
                # Faltando campos obrigatórios
            )


class ModelMethodsTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )

    def test_modelo_is_boost_ativo(self):
        """Testa método is_boost_ativo"""
        from models.models import PlanoAnuncio
        from django.utils import timezone
        from datetime import timedelta
        
        # Sem plano ativo
        self.assertFalse(self.modelo.is_boost_ativo)
        
        # Com plano ativo
        plano = PlanoAnuncio.objects.create(
            nome='Plano Teste',
            tipo='basico',
            duracao_dias=30,
            preco=50.00
        )
        self.modelo.plano_ativo = plano
        self.modelo.plano_data_fim = timezone.now() + timedelta(days=30)
        self.modelo.save()
        self.assertTrue(self.modelo.is_boost_ativo)

    def test_modelo_get_total_views(self):
        """Testa método get_total_views"""
        # Sem visualizações
        self.assertEqual(self.modelo.get_total_views(), 0)
        
        # Com visualizações (se implementado)
        # Este teste pode ser expandido quando a funcionalidade for implementada

    def test_modelo_get_monthly_views(self):
        """Testa método get_monthly_views"""
        # Sem visualizações mensais
        self.assertEqual(self.modelo.get_monthly_views(), 0)
        
        # Com visualizações mensais (se implementado)
        # Este teste pode ser expandido quando a funcionalidade for implementada
