#!/usr/bin/env python
"""
Script de migração para importar dados do projeto PHP antigo para o Django
"""
import os
import sys
import django
import shutil
from datetime import datetime
from decimal import Decimal

# Configurar Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')
django.setup()

from django.contrib.auth.models import User
from django.core.files import File
from django.core.files.temp import NamedTemporaryFile
from django.utils import timezone
from models.models import Modelo, FotoModelo, Servico, Banner, VisualizacaoModelo

# Configurações
OLD_UPLOADS_PATH = '../allure_antigo/uploads'
NEW_MEDIA_PATH = 'media'

# Mapeamento de categorias do sistema antigo para o novo
CATEGORIA_MAPPING = {
    'anuncio': 'JOB',
    'vendas_conteudos': 'FAN',
    'camera_ao_vivo': 'CAM',
}

# Mapeamento de etnias
ETNIA_MAPPING = {
    'morena': 'parda',
    'loira': 'branca',
    'negra': 'negra',
    'mestica': 'parda',
    'asiatica': 'asiatica',
    'indigena': 'indigena',
}

# Mapeamento de local de atendimento
LOCAL_MAPPING = {
    'proprio': 'local_proprio',
    'hoteis': 'hotel',
    'motel': 'motel',
    'domicilio': 'residencia_cliente',
}

def create_user_for_modelo(modelo_data):
    """Cria um usuário Django para a modelo"""
    email = modelo_data['email']
    username = f"modelo_{modelo_data['id']}"
    
    # Verificar se o usuário já existe
    if User.objects.filter(email=email).exists():
        return User.objects.get(email=email)
    
    # Criar novo usuário
    user = User.objects.create_user(
        username=username,
        email=email,
        password='temp_password_123',  # Senha temporária
        first_name=modelo_data['nome'],
        last_name=modelo_data['sobrenome']
    )
    
    return user

def migrate_modelos():
    """Migra os dados das modelos"""
    print("🔄 Migrando modelos...")
    
    # Dados das modelos do sistema antigo (extraídos do SQL)
    modelos_data = [
        {
            'id': 56, 'nome': 'Larissa', 'sobrenome': 'Soares', 'nome_exibicao': 'Larissa',
            'idade': 23, 'cpf': '258.296.548-08', 'etnia': 'morena', 'telefone': '(11) 98403-2639',
            'email': 'larissa2025@gmail.com', 'data_nascimento': '2001-10-29', 'estado': 'SP', 'cidade': 'Sabino',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio',
            'foto_capa': 'capa_686c386725d63.jpg', 'foto_perfil': 'perfil_685ea4da43850.jpg',
            'status': 'aprovado', 'plano': 'premium_pendente'
        },
        {
            'id': 57, 'nome': 'Carol', 'sobrenome': 'Ramos', 'nome_exibicao': 'Carol',
            'idade': 23, 'cpf': '258.296.548-10', 'etnia': 'loira', 'telefone': '(11) 98403-2738',
            'email': 'carol2025@gmail.com', 'data_nascimento': '2001-10-29', 'estado': 'SP', 'cidade': 'Sales',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio,massagens_eroticas',
            'foto_capa': 'capa_685ea798e6fe8.jpg', 'foto_perfil': 'perfil_685ea798e6654.jpg',
            'status': 'aprovado', 'plano': 'free'
        },
        {
            'id': 58, 'nome': 'Ana', 'sobrenome': 'Joel', 'nome_exibicao': 'Aninha',
            'idade': 22, 'cpf': '258.296.555-55', 'etnia': 'morena,mestica', 'telefone': '(11) 98433-7666',
            'email': 'aninha2025@gmail.com', 'data_nascimento': '2002-10-29', 'estado': 'SP', 'cidade': 'Sales Oliveira',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio',
            'foto_capa': 'capa_685eb11ce9516.jpg', 'foto_perfil': 'perfil_685eb11ce89c5.jpg',
            'status': 'aprovado', 'plano': 'free'
        },
        {
            'id': 59, 'nome': 'Brisa', 'sobrenome': 'medeiros', 'nome_exibicao': 'Brisa e Flora',
            'idade': 22, 'cpf': '258.276.548-06', 'etnia': 'morena', 'telefone': '(11) 98744-6777',
            'email': 'brisa2025@gmail.com', 'data_nascimento': '2002-10-29', 'estado': 'SP', 'cidade': 'Sabino',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio,massagens_eroticas',
            'foto_capa': 'capa_685eb45aae863.jpg', 'foto_perfil': 'perfil_685eb45aad5a1.jpg',
            'status': 'aprovado', 'plano': 'free'
        },
        {
            'id': 60, 'nome': 'Nice', 'sobrenome': 'Mota', 'nome_exibicao': 'Nice Flora Brisa e Nat',
            'idade': 22, 'cpf': '254.296.547-06', 'etnia': 'morena', 'telefone': '(11) 87554-4444',
            'email': 'nice2025@gmail.com', 'data_nascimento': '2002-10-29', 'estado': 'SP', 'cidade': 'Sabino',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio,hoteis,outro', 'categoria': 'anuncio,sexting',
            'foto_capa': 'capa_685eb896500a1.jpg', 'foto_perfil': 'perfil_685eb8964fa82.jpg',
            'status': 'aprovado', 'plano': 'free'
        },
        {
            'id': 61, 'nome': 'Cacau', 'sobrenome': 'Lorna', 'nome_exibicao': 'Cacau',
            'idade': 22, 'cpf': '258.222.444-33', 'etnia': 'negra', 'telefone': '(11) 98855-6677',
            'email': 'cacau2025@gmail.com', 'data_nascimento': '2002-10-29', 'estado': 'SP', 'cidade': 'Salesópolis',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio',
            'foto_capa': 'capa_685ec87703e51.jpg', 'foto_perfil': 'perfil_685ec877036b2.jpg',
            'status': 'aprovado', 'plano': 'free'
        },
        {
            'id': 62, 'nome': 'Jenifer', 'sobrenome': 'Prax', 'nome_exibicao': 'Jenifer',
            'idade': 22, 'cpf': '228.233.333-03', 'etnia': 'morena', 'telefone': '(11) 98888-5555',
            'email': 'jenifer2025@gmail.com', 'data_nascimento': '2002-10-20', 'estado': 'SC', 'cidade': 'Saltinho',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio',
            'foto_capa': 'capa_685ed0c07ead9.jpg', 'foto_perfil': 'perfil_685ed0c07df7d.jpg',
            'status': 'aprovado', 'plano': 'free'
        },
        {
            'id': 63, 'nome': 'Natalia', 'sobrenome': 'Lemos', 'nome_exibicao': 'Naty Brisa e Flora',
            'idade': 22, 'cpf': '245.566.666-67', 'etnia': 'morena', 'telefone': '(11) 98766-4455',
            'email': 'naty2025@gmail.com', 'data_nascimento': '2002-10-29', 'estado': 'SC', 'cidade': 'Salete',
            'descricao': 'My oral is deep and wet...', 'local_atendimento': 'proprio', 'categoria': 'anuncio,massagens_eroticas',
            'foto_capa': 'capa_685ed592dcd84.jpg', 'foto_perfil': 'perfil_685ed592dc1e5.jpg',
            'status': 'aprovado', 'plano': 'premium_pendente'
        },
        {
            'id': 64, 'nome': 'Flavia Bernardes', 'sobrenome': 'Tamaio', 'nome_exibicao': 'Pamela Pantera',
            'idade': 27, 'cpf': '466.147.018-43', 'etnia': 'morena', 'telefone': '(11) 98043-5812',
            'email': 'flaviatamayooo@gmail.com', 'data_nascimento': '1997-12-10', 'estado': 'SP', 'cidade': 'São Paulo',
            'descricao': 'olá meu amor, eu sou a Pamela Pantera...', 'local_atendimento': 'proprio,hoteis,domicilio,outro', 
            'categoria': 'anuncio,camera_ao_vivo,vendas_conteudos,massagens_eroticas,videochamadas,sexting',
            'foto_capa': 'capa_6862ac19e8583.jpg', 'foto_perfil': 'perfil_6862ac19d3156.jpg',
            'status': 'aprovado', 'plano': 'premium_pendente'
        },
        {
            'id': 65, 'nome': 'Clezia Cristina', 'sobrenome': 'Dias', 'nome_exibicao': 'Bia mello',
            'idade': 47, 'cpf': '016.292.361-95', 'etnia': 'loira', 'telefone': '(19) 98710-6844',
            'email': 'tininhacometa@hotmail.com', 'data_nascimento': '1977-09-19', 'estado': 'SP', 'cidade': 'Piracicaba',
            'descricao': 'Olá meus amores, me chamo Bia Mello...', 'local_atendimento': 'proprio,hoteis', 
            'categoria': 'anuncio,vendas_conteudos,videochamadas',
            'foto_capa': 'capa_686585bfc144e.jpg', 'foto_perfil': 'perfil_686585bf7e7a3.jpg',
            'status': 'aprovado', 'plano': 'free'
        }
    ]
    
    modelos_criadas = []
    
    for modelo_data in modelos_data:
        try:
            # Verificar se a modelo já existe
            if Modelo.objects.filter(user__email=modelo_data['email']).exists():
                print(f"⚠️  Modelo {modelo_data['nome_exibicao']} já existe, pulando...")
                continue
            
            # Criar usuário
            user = create_user_for_modelo(modelo_data)
            
            # Processar categorias
            categorias_antigas = modelo_data['categoria'].split(',')
            categoria_principal = CATEGORIA_MAPPING.get(categorias_antigas[0], 'JOB')
            
            # Processar etnia
            etnia_antiga = modelo_data['etnia'].split(',')[0]
            etnia_nova = ETNIA_MAPPING.get(etnia_antiga, 'outra')
            
            # Processar local de atendimento
            locais_antigos = modelo_data['local_atendimento'].split(',')
            local_principal = LOCAL_MAPPING.get(locais_antigos[0], 'local_proprio')
            
            # Criar modelo
            modelo = Modelo.objects.create(
                user=user,
                nome_exibicao=modelo_data['nome_exibicao'],
                idade=modelo_data['idade'],
                estado=modelo_data['estado'],
                cidade=modelo_data['cidade'],
                etnia=etnia_nova,
                categoria=categoria_principal,
                tipo_ensaio='solo',  # Padrão
                local_atendimento=local_principal,
                sobre_mim=modelo_data['descricao'],
                telefone=modelo_data['telefone'],
                whatsapp=modelo_data['telefone'],
                email_contato=modelo_data['email'],
                status=modelo_data['status'],
                premium=(modelo_data['plano'] == 'premium_pendente'),
                data_cadastro=timezone.now()
            )
            
            modelos_criadas.append(modelo)
            print(f"✅ Modelo {modelo.nome_exibicao} criada com sucesso!")
            
        except Exception as e:
            print(f"❌ Erro ao criar modelo {modelo_data['nome_exibicao']}: {e}")
    
    print(f"🎉 {len(modelos_criadas)} modelos migradas com sucesso!")
    return modelos_criadas

def migrate_photos():
    """Migra as fotos das modelos"""
    print("🔄 Migrando fotos...")
    
    # Dados das fotos (extraídos do SQL)
    fotos_data = [
        # Modelo 56 - Larissa
        {'modelo_id': 56, 'tipo': 'perfil', 'arquivo': '56/perfil_685ea4da43850.jpg'},
        {'modelo_id': 56, 'tipo': 'galeria', 'arquivo': '56/foto_685ea5a876528.jpg'},
        {'modelo_id': 56, 'tipo': 'galeria', 'arquivo': '56/foto_685ea5c28882e.jpg'},
        {'modelo_id': 56, 'tipo': 'galeria', 'arquivo': '56/foto_685ea5cf71d35.jpg'},
        {'modelo_id': 56, 'tipo': 'galeria', 'arquivo': '56/foto_685ea5dc682d5.jpg'},
        
        # Modelo 57 - Carol
        {'modelo_id': 57, 'tipo': 'perfil', 'arquivo': '57/perfil_685ea798e6654.jpg'},
        {'modelo_id': 57, 'tipo': 'galeria', 'arquivo': '57/foto_685ea7d6d4118.jpg'},
        {'modelo_id': 57, 'tipo': 'galeria', 'arquivo': '57/foto_685ea7e47c13a.jpg'},
        
        # Modelo 58 - Aninha
        {'modelo_id': 58, 'tipo': 'perfil', 'arquivo': '58/perfil_685eb11ce89c5.jpg'},
        {'modelo_id': 58, 'tipo': 'galeria', 'arquivo': '58/foto_685eb19238e35.jpg'},
        {'modelo_id': 58, 'tipo': 'galeria', 'arquivo': '58/foto_685eb1a6971a1.jpg'},
        
        # Modelo 59 - Brisa
        {'modelo_id': 59, 'tipo': 'perfil', 'arquivo': '59/perfil_685eb45aad5a1.jpg'},
        {'modelo_id': 59, 'tipo': 'galeria', 'arquivo': '59/foto_685eb49d60166.jpg'},
        {'modelo_id': 59, 'tipo': 'galeria', 'arquivo': '59/foto_685eb4a9c3761.jpg'},
        
        # Modelo 60 - Nice
        {'modelo_id': 60, 'tipo': 'perfil', 'arquivo': '60/perfil_685eb8964fa82.jpg'},
        {'modelo_id': 60, 'tipo': 'galeria', 'arquivo': '60/foto_685eb8f2f3be4.jpg'},
        {'modelo_id': 60, 'tipo': 'galeria', 'arquivo': '60/foto_685eb91356e59.jpg'},
        
        # Modelo 61 - Cacau
        {'modelo_id': 61, 'tipo': 'perfil', 'arquivo': '61/perfil_685ec877036b2.jpg'},
        {'modelo_id': 61, 'tipo': 'galeria', 'arquivo': '61/foto_685ec8ba93909.jpg'},
        {'modelo_id': 61, 'tipo': 'galeria', 'arquivo': '61/foto_685ec91045747.jpg'},
        
        # Modelo 62 - Jenifer
        {'modelo_id': 62, 'tipo': 'perfil', 'arquivo': '62/perfil_685ed0c07df7d.jpg'},
        {'modelo_id': 62, 'tipo': 'galeria', 'arquivo': '62/foto_685ed11a6a7b9.jpg'},
        {'modelo_id': 62, 'tipo': 'galeria', 'arquivo': '62/foto_685ed13948c1b.jpg'},
        
        # Modelo 63 - Naty
        {'modelo_id': 63, 'tipo': 'perfil', 'arquivo': '63/perfil_685ed592dc1e5.jpg'},
        {'modelo_id': 63, 'tipo': 'galeria', 'arquivo': '63/foto_685ed5ce9e1b9.jpg'},
        {'modelo_id': 63, 'tipo': 'galeria', 'arquivo': '63/foto_685ed5de29853.jpg'},
        
        # Modelo 64 - Pamela Pantera
        {'modelo_id': 64, 'tipo': 'perfil', 'arquivo': '64/perfil_6862ac19d3156.jpg'},
        {'modelo_id': 64, 'tipo': 'galeria', 'arquivo': '64/foto_6862dd2a9c829.jpg'},
        {'modelo_id': 64, 'tipo': 'galeria', 'arquivo': '64/foto_6862dd78b2807.jpg'},
        
        # Modelo 65 - Bia Mello
        {'modelo_id': 65, 'tipo': 'perfil', 'arquivo': '65/perfil_686585bf7e7a3.jpg'},
    ]
    
    fotos_criadas = 0
    
    for foto_data in fotos_data:
        try:
            # Encontrar a modelo pelo ID do usuário
            try:
                user = User.objects.get(username=f"modelo_{foto_data['modelo_id']}")
                modelo = Modelo.objects.get(user=user)
            except (User.DoesNotExist, Modelo.DoesNotExist):
                print(f"⚠️  Modelo {foto_data['modelo_id']} não encontrada, pulando foto...")
                continue
            
            # Verificar se a foto já existe
            if FotoModelo.objects.filter(modelo=modelo, imagem__contains=os.path.basename(foto_data['arquivo'])).exists():
                continue
            
            # Caminho do arquivo antigo
            old_file_path = os.path.join(OLD_UPLOADS_PATH, 'modelos', foto_data['arquivo'])
            
            if not os.path.exists(old_file_path):
                print(f"⚠️  Arquivo não encontrado: {old_file_path}")
                continue
            
            # Copiar arquivo para o novo local
            new_filename = f"modelo_{modelo.id}_{os.path.basename(foto_data['arquivo'])}"
            new_file_path = os.path.join(NEW_MEDIA_PATH, 'modelos', 'fotos', new_filename)
            
            # Criar diretório se não existir
            os.makedirs(os.path.dirname(new_file_path), exist_ok=True)
            
            # Copiar arquivo
            shutil.copy2(old_file_path, new_file_path)
            
            # Criar registro no banco
            categoria_foto = 'galeria' if foto_data['tipo'] == 'galeria' else 'rosto'
            
            foto = FotoModelo.objects.create(
                modelo=modelo,
                imagem=f'modelos/fotos/{new_filename}',
                categoria=categoria_foto,
                ordem=fotos_criadas
            )
            
            fotos_criadas += 1
            
        except Exception as e:
            print(f"❌ Erro ao migrar foto {foto_data['arquivo']}: {e}")
    
    print(f"🎉 {fotos_criadas} fotos migradas com sucesso!")

def migrate_services():
    """Migra os serviços das modelos"""
    print("🔄 Migrando serviços...")
    
    # Dados dos serviços (extraídos do SQL)
    servicos_data = [
        {'modelo_id': 56, 'descricao': '1 hora', 'preco': 500.00},
        {'modelo_id': 57, 'descricao': '1 hora', 'preco': 400.00},
        {'modelo_id': 58, 'descricao': '1 hora', 'preco': 750.00},
        {'modelo_id': 59, 'descricao': '1 hora', 'preco': 2000.00},
        {'modelo_id': 60, 'descricao': '1 hora', 'preco': 8000.00},
        {'modelo_id': 61, 'descricao': '1 hora', 'preco': 700.00},
        {'modelo_id': 62, 'descricao': '1 hora', 'preco': 500.00},
        {'modelo_id': 63, 'descricao': '1 hora nós 3', 'preco': 1500.00},
        {'modelo_id': 64, 'descricao': '1 hora', 'preco': 400.00},
        {'modelo_id': 65, 'descricao': '1 hora', 'preco': 250.00},
    ]
    
    servicos_criados = 0
    
    for servico_data in servicos_data:
        try:
            # Encontrar a modelo pelo ID do usuário
            try:
                user = User.objects.get(username=f"modelo_{servico_data['modelo_id']}")
                modelo = Modelo.objects.get(user=user)
            except (User.DoesNotExist, Modelo.DoesNotExist):
                print(f"⚠️  Modelo {servico_data['modelo_id']} não encontrada, pulando serviço...")
                continue
            
            # Verificar se o serviço já existe
            if Servico.objects.filter(modelo=modelo, descricao=servico_data['descricao']).exists():
                continue
            
            # Criar serviço
            servico = Servico.objects.create(
                modelo=modelo,
                descricao=servico_data['descricao'],
                preco=Decimal(str(servico_data['preco'])),
                duracao='1 hora',
                ativo=True
            )
            
            servicos_criados += 1
            
        except Exception as e:
            print(f"❌ Erro ao criar serviço para modelo {servico_data['modelo_id']}: {e}")
    
    print(f"🎉 {servicos_criados} serviços migrados com sucesso!")

def migrate_banners():
    """Migra os banners"""
    print("🔄 Migrando banners...")
    
    # Dados dos banners (extraídos do SQL)
    banners_data = [
        {
            'titulo': '1 - pamela pantera',
            'imagem': '68640b3f60890.JPG',
            'link': 'https://allure-club.com/modelo_detalhes?id=64',
            'ordem': 0
        },
        {
            'titulo': 'bia mello',
            'imagem': '686813840937a.JPG',
            'link': 'https://allure-club.com/modelo_detalhes?id=65',
            'ordem': 1
        }
    ]
    
    banners_criados = 0
    
    for banner_data in banners_data:
        try:
            # Verificar se o banner já existe
            if Banner.objects.filter(titulo=banner_data['titulo']).exists():
                continue
            
            # Caminho do arquivo antigo
            old_file_path = os.path.join(OLD_UPLOADS_PATH, 'banners', banner_data['imagem'])
            
            if not os.path.exists(old_file_path):
                print(f"⚠️  Arquivo de banner não encontrado: {old_file_path}")
                continue
            
            # Copiar arquivo para o novo local
            new_filename = f"banner_{banner_data['ordem']}_{banner_data['imagem']}"
            new_file_path = os.path.join(NEW_MEDIA_PATH, 'banners', new_filename)
            
            # Criar diretório se não existir
            os.makedirs(os.path.dirname(new_file_path), exist_ok=True)
            
            # Copiar arquivo
            shutil.copy2(old_file_path, new_file_path)
            
            # Criar banner
            banner = Banner.objects.create(
                titulo=banner_data['titulo'],
                imagem=f'banners/{new_filename}',
                link=banner_data['link'],
                ordem=banner_data['ordem'],
                ativo=True
            )
            
            banners_criados += 1
            
        except Exception as e:
            print(f"❌ Erro ao criar banner {banner_data['titulo']}: {e}")
    
    print(f"🎉 {banners_criados} banners migrados com sucesso!")

def main():
    """Função principal de migração"""
    print("🚀 Iniciando migração de dados do projeto antigo...")
    print("=" * 50)
    
    # Criar diretórios necessários
    os.makedirs(os.path.join(NEW_MEDIA_PATH, 'modelos', 'fotos'), exist_ok=True)
    os.makedirs(os.path.join(NEW_MEDIA_PATH, 'banners'), exist_ok=True)
    
    # Executar migrações
    try:
        modelos = migrate_modelos()
        migrate_photos()
        migrate_services()
        migrate_banners()
        
        print("\n" + "=" * 50)
        print("🎉 Migração concluída com sucesso!")
        print(f"📊 Resumo:")
        print(f"   - Modelos migradas: {len(modelos)}")
        print(f"   - Fotos migradas: {FotoModelo.objects.count()}")
        print(f"   - Serviços migrados: {Servico.objects.count()}")
        print(f"   - Banners migrados: {Banner.objects.count()}")
        print("\n⚠️  IMPORTANTE:")
        print("   - As senhas foram definidas como 'temp_password_123'")
        print("   - As modelos devem redefinir suas senhas no primeiro login")
        print("   - Verifique se todos os arquivos de mídia foram copiados corretamente")
        
    except Exception as e:
        print(f"❌ Erro durante a migração: {e}")
        return False
    
    return True

if __name__ == '__main__':
    main() 