#!/usr/bin/env python3
"""
Script para executar o servidor de desenvolvimento Django + SASS Watch
"""
import os
import sys
import time
import subprocess
import threading
from pathlib import Path

def run_django_server():
    """Executa o servidor Django"""
    try:
        print("🚀 Iniciando servidor Django na porta 8010...")
        subprocess.run([
            sys.executable, 'manage.py', 'runserver', '8010'
        ], check=True)
    except subprocess.CalledProcessError as e:
        print(f"❌ Erro ao executar servidor Django: {e}")
    except KeyboardInterrupt:
        print("\n🛑 Servidor Django parado")

def run_sass_watch():
    """Executa o monitoramento SASS"""
    try:
        print("🎨 Iniciando monitoramento SASS...")
        subprocess.run([
            sys.executable, 'watch_sass.py'
        ], check=True)
    except subprocess.CalledProcessError as e:
        print(f"❌ Erro ao executar SASS watch: {e}")
    except KeyboardInterrupt:
        print("\n🛑 Monitoramento SASS parado")

def main():
    """Função principal"""
    project_root = Path(__file__).parent
    
    # Verificar se estamos no diretório correto
    if not (project_root / 'manage.py').exists():
        print("❌ Erro: Este script deve ser executado no diretório raiz do projeto Django")
        sys.exit(1)
    
    # Verificar se o ambiente virtual existe
    if not (project_root / 'venv').exists():
        print("❌ Erro: Ambiente virtual não encontrado")
        print("   Execute: python3 -m venv venv")
        sys.exit(1)
    
    print("🚀 Iniciando servidor de desenvolvimento Allure...")
    print("=" * 50)
    print("📱 Django Server: http://localhost:8010")
    print("🎨 SASS Watch: Ativo")
    print("=" * 50)
    print("Pressione Ctrl+C para parar ambos os serviços")
    print()
    
    # Criar threads para executar Django e SASS watch simultaneamente
    django_thread = threading.Thread(target=run_django_server, daemon=True)
    sass_thread = threading.Thread(target=run_sass_watch, daemon=True)
    
    try:
        # Iniciar ambos os threads
        django_thread.start()
        time.sleep(2)  # Pequena pausa para Django inicializar
        sass_thread.start()
        
        # Manter o script rodando
        while True:
            time.sleep(1)
            
    except KeyboardInterrupt:
        print("\n🛑 Parando servidor de desenvolvimento...")
        sys.exit(0)

if __name__ == '__main__':
    main()
