from django.shortcuts import render, get_object_or_404, redirect
from django.http import JsonResponse
from django.core.paginator import Paginator
from django.db.models import Q, Count, Case, When, IntegerField, Avg
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.utils import timezone
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods
from models.models import Banner, Modelo, FotoModelo, Servico, VisualizacaoModelo, Favorito, Avaliacao, Comentario, VideoModelo
import json


def home(request):
    """View da página inicial - apenas modelos com planos ativos"""
    # Buscar banners ativos ordenados
    banners = Banner.objects.filter(ativo=True).order_by('ordem')
    
    # Buscar modelos com planos ativos e perfil completo
    modelos_base = Modelo.objects.filter(
        status__in=['aprovado', 'ativo'],
        plano_ativo__isnull=False,
        plano_data_fim__gt=timezone.now()
    ).exclude(
        Q(foto_perfil='') | Q(foto_perfil__isnull=True)
    ).exclude(
        Q(sobre_mim='') | Q(sobre_mim__isnull=True)
    ).exclude(
        Q(telefone='') & Q(whatsapp='') & Q(telegram='') & Q(link_contato='') & Q(email_contato='')
    ).select_related('plano_ativo').prefetch_related('boost_usage')
    
    # Modelos com boost ativo (primeiro)
    modelos_com_boost = modelos_base.filter(
        boost_usage__data_expiracao__gt=timezone.now(),
        boost_usage__usado=True
    ).distinct()
    
    # Modelos sem boost ativo
    modelos_sem_boost = modelos_base.exclude(
        boost_usage__data_expiracao__gt=timezone.now(),
        boost_usage__usado=True
    )
    
    # Aplicar filtros da URL
    cidade = request.GET.get('cidade')
    if cidade:
        modelos_base = modelos_base.filter(cidade__icontains=cidade)
    
    estado = request.GET.get('estado')
    if estado:
        modelos_base = modelos_base.filter(estado=estado)
    
    categoria_servico = request.GET.getlist('categoria_servico')
    if categoria_servico:
        modelos_base = modelos_base.filter(categorias__id__in=categoria_servico).distinct()
    
    local_atendimento = request.GET.getlist('local_atendimento')
    if local_atendimento:
        modelos_base = modelos_base.filter(local_atendimento__in=local_atendimento)
    
    idade_min = request.GET.get('idade_min')
    if idade_min:
        modelos_base = modelos_base.filter(idade__gte=idade_min)
    
    idade_max = request.GET.get('idade_max')
    if idade_max:
        modelos_base = modelos_base.filter(idade__lte=idade_max)
    
    etnia = request.GET.getlist('etnia')
    if etnia:
        modelos_base = modelos_base.filter(etnia__in=etnia)
    
    categoria_foto = request.GET.getlist('categoria_foto')
    if categoria_foto:
        modelos_base = modelos_base.filter(fotos__categoria__in=categoria_foto).distinct()
    
    # Aplicar ordenação baseada nos parâmetros da URL
    ordenacao = request.GET.get('ordenacao', 'recentes')
    from django.db.models import Case, When, Value, BooleanField
    
    modelos_destaque = modelos_base.annotate(
        tem_boost=Case(
            When(
                boost_usage__data_expiracao__gt=timezone.now(),
                boost_usage__usado=True,
                then=Value(True)
            ),
            default=Value(False),
            output_field=BooleanField()
        )
    )
    
    # Aplicar ordenação específica
    if ordenacao == 'mais_vistas_total':
        # Para ordenar por total de visualizações, vamos usar uma abordagem diferente
        # Primeiro vamos buscar todos os modelos e depois ordenar em Python
        modelos_destaque = list(modelos_destaque)
        modelos_destaque.sort(key=lambda x: (x.tem_boost, x.get_total_views(), x.data_atualizacao), reverse=True)
        modelos_destaque = modelos_destaque[:8]
    elif ordenacao == 'mais_vistas_mes':
        # Para ordenar por visualizações do mês
        modelos_destaque = list(modelos_destaque)
        modelos_destaque.sort(key=lambda x: (x.tem_boost, x.get_monthly_views(), x.data_atualizacao), reverse=True)
        modelos_destaque = modelos_destaque[:8]
    else:  # recentes
        modelos_destaque = modelos_destaque.order_by('-tem_boost', '-data_atualizacao')[:8]
    
    # Modelos recentes (mesmo critério)
    modelos_recentes = modelos_base.annotate(
        tem_boost=Case(
            When(
                boost_usage__data_expiracao__gt=timezone.now(),
                boost_usage__usado=True,
                then=Value(True)
            ),
            default=Value(False),
            output_field=BooleanField()
        )
    ).order_by('-tem_boost', '-data_cadastro')[:12]
    
    # Buscar categorias de serviços que têm modelos cadastradas
    from models.models import Categoria
    categorias_com_modelos = Categoria.objects.filter(
        modelo__status__in=['aprovado', 'ativo']
    ).distinct().order_by('ordem', 'nome')
    
    context = {
        'banners': banners,
        'modelos_destaque': modelos_destaque,
        'modelos_recentes': modelos_recentes,
        'apenas_pagantes': True,  # Flag para mostrar que só modelos pagantes aparecem
        'ordenacao_atual': ordenacao,
        'categorias_servico': categorias_com_modelos,
        'filtros_aplicados': {
            'cidade': cidade,
            'estado': estado,
            'categoria_servico': categoria_servico,
            'local_atendimento': local_atendimento,
            'idade_min': idade_min,
            'idade_max': idade_max,
            'etnia': etnia,
            'categoria_foto': categoria_foto,
        },
    }
    
    return render(request, 'core/home.html', context)


def modelo_detail(request, modelo_slug):
    """View do perfil individual da modelo"""
    modelo = get_object_or_404(Modelo, slug=modelo_slug, status__in=['aprovado', 'ativo'])
    
    # Registrar visualização
    ip_address = get_client_ip(request)
    user_agent = request.META.get('HTTP_USER_AGENT', '')
    
    # Evitar múltiplas visualizações do mesmo IP em pouco tempo
    recent_view = VisualizacaoModelo.objects.filter(
        modelo=modelo,
        ip_address=ip_address,
        data_visualizacao__gte=timezone.now() - timezone.timedelta(minutes=30)
    ).exists()
    
    if not recent_view:
        VisualizacaoModelo.objects.create(
            modelo=modelo,
            ip_address=ip_address,
            user_agent=user_agent
        )
    
    # Buscar fotos da modelo
    fotos_galeria = modelo.fotos.filter(categoria='galeria').order_by('ordem')
    fotos_seletas = {}
    for categoria in ['seios', 'frente', 'bumbum', 'rosto']:
        fotos_seletas[categoria] = modelo.fotos.filter(categoria=categoria).first()
    
    # Buscar serviços
    servicos = modelo.servicos.filter(ativo=True).order_by('preco')
    
    # Buscar vídeos
    videos = modelo.videos.filter(ativo=True).order_by('-data_upload')[:6]
    
    # Buscar avaliações aprovadas
    avaliacoes = modelo.avaliacoes.filter(aprovado=True).order_by('-data_avaliacao')[:5]
    
    # Buscar comentários aprovados
    comentarios = modelo.comentarios.filter(aprovado=True).order_by('-data_comentario')[:3]
    
    # Calcular média das avaliações
    media_avaliacoes = modelo.avaliacoes.filter(aprovado=True).aggregate(
        media=Avg('nota')
    )['media'] or 0
    
    # Verificar se está nos favoritos do usuário
    is_favorito = False
    if request.user.is_authenticated:
        is_favorito = Favorito.objects.filter(user=request.user, modelo=modelo).exists()
    
    context = {
        'modelo': modelo,
        'fotos_galeria': fotos_galeria,
        'fotos_seletas': fotos_seletas,
        'servicos': servicos,
        'videos': videos,
        'avaliacoes': avaliacoes,
        'comentarios': comentarios,
        'media_avaliacoes': round(media_avaliacoes, 1),
        'total_avaliacoes': modelo.avaliacoes.filter(aprovado=True).count(),
        'is_favorito': is_favorito,
        'total_views': modelo.get_total_views(),
        'monthly_views': modelo.get_monthly_views(),
    }
    
    return render(request, 'core/modelo_detail.html', context)


@login_required
@csrf_exempt
@require_http_methods(["POST"])
def toggle_favorito(request):
    """Toggle favorito via AJAX"""
    try:
        data = json.loads(request.body)
        modelo_id = data.get('modelo_id')
        
        modelo = get_object_or_404(Modelo, id=modelo_id, status__in=['aprovado', 'ativo'])
        favorito, created = Favorito.objects.get_or_create(
            user=request.user,
            modelo=modelo
        )
        
        if not created:
            favorito.delete()
            is_favorito = False
        else:
            is_favorito = True
        
        return JsonResponse({
            'success': True,
            'is_favorito': is_favorito,
            'total_favoritos': modelo.favoritado_por.count()
        })
        
    except Exception as e:
        return JsonResponse({
            'success': False,
            'error': str(e)
        })


def get_cidades(request):
    """API para buscar cidades por estado"""
    estado = request.GET.get('estado')
    if estado:
        cidades = Modelo.objects.filter(
            estado=estado,
            status='aprovado'
        ).values_list('cidade', flat=True).distinct()
        return JsonResponse({
            'cidades': sorted(list(cidades))
        })
    return JsonResponse({'cidades': []})


def get_cidades_com_modelos(request):
    """API para buscar cidades organizadas por estado que têm modelos cadastrados"""
    from collections import defaultdict
    
    # Buscar modelos aprovados com perfil completo
    modelos = Modelo.objects.filter(
        status__in=['aprovado', 'ativo']
    ).exclude(
        Q(foto_perfil='') | Q(foto_perfil__isnull=True)
    ).exclude(
        Q(sobre_mim='') | Q(sobre_mim__isnull=True)
    ).exclude(
        Q(telefone='') & Q(whatsapp='') & Q(telegram='') & Q(link_contato='') & Q(email_contato='')
    ).values('estado', 'cidade').distinct()
    
    # Organizar por estado
    cidades_por_estado = defaultdict(list)
    for modelo in modelos:
        if modelo['cidade'] and modelo['estado']:
            cidades_por_estado[modelo['estado']].append(modelo['cidade'])
    
    # Ordenar cidades dentro de cada estado
    for estado in cidades_por_estado:
        cidades_por_estado[estado] = sorted(list(set(cidades_por_estado[estado])))
    
    return JsonResponse(dict(cidades_por_estado))


def verificacao_idade(request):
    """View para verificação de idade"""
    if request.method == 'POST':
        confirmacao = request.POST.get('confirmacao')
        if confirmacao == 'sim':
            request.session['idade_verificada'] = True
            return redirect('core:home')
        else:
            return redirect('https://www.google.com')
    
    return render(request, 'core/verificacao_idade.html')


def get_client_ip(request):
    """Função auxiliar para obter o IP do cliente"""
    x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
    if x_forwarded_for:
        ip = x_forwarded_for.split(',')[0]
    else:
        ip = request.META.get('REMOTE_ADDR')
    return ip


def seja_modelo(request):
    """View da página 'Seja Nossa Modelo'"""
    return render(request, 'core/seja_modelo.html')


def termos(request):
    """View da página de termos de uso"""
    return render(request, 'core/termos.html')


def politica_privacidade(request):
    """View da página de política de privacidade"""
    return render(request, 'core/politica_privacidade.html')


def politica_cookies(request):
    """View da página de política de cookies"""
    return render(request, 'core/politica_cookies.html')


@login_required
@require_http_methods(["POST"])
def ativar_boost(request):
    """View para ativar boost da modelo"""
    try:
        modelo = request.user.modelo
    except:
        return JsonResponse({'success': False, 'message': 'Acesso negado.'})
    
    # Tentar ativar o boost
    sucesso, mensagem = modelo.usar_boost()
    
    return JsonResponse({
        'success': sucesso,
        'message': mensagem
    })

