from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from models.models import Modelo, PlanoAnuncio
from payments.models import TransacaoPagamento, PlanoAnuncioPagamento
from payments.services import PlanoAnuncioService, mercado_pago_service
from django.utils import timezone


class Command(BaseCommand):
    help = 'Testa o fluxo completo de pagamento de planos'

    def add_arguments(self, parser):
        parser.add_argument(
            '--criar-modelo-teste',
            action='store_true',
            help='Cria uma modelo de teste para os testes'
        )
        parser.add_argument(
            '--testar-selecao-plano',
            action='store_true',
            help='Testa a seleção e criação de pagamento'
        )
        parser.add_argument(
            '--simular-pagamento-aprovado',
            action='store_true',
            help='Simula pagamento aprovado e ativação do plano'
        )
        parser.add_argument(
            '--teste-completo',
            action='store_true',
            help='Executa todos os testes em sequência'
        )

    def handle(self, *args, **options):
        self.stdout.write(
            self.style.SUCCESS('🚀 Iniciando testes do fluxo de pagamento...')
        )
        
        if options['criar_modelo_teste'] or options['teste_completo']:
            self.testar_criacao_modelo()
        
        if options['testar_selecao_plano'] or options['teste_completo']:
            self.testar_selecao_plano()
        
        if options['simular_pagamento_aprovado'] or options['teste_completo']:
            self.simular_pagamento_aprovado()
        
        if not any(options.values()):
            self.stdout.write(
                self.style.WARNING('Use --help para ver as opções disponíveis')
            )
    
    def testar_criacao_modelo(self):
        """Testa a criação de uma modelo de teste"""
        self.stdout.write('\n👩 Teste 1: Criando modelo de teste...')
        
        try:
            # Criar usuário de teste
            user, created = User.objects.get_or_create(
                username='modelo_teste',
                defaults={
                    'email': 'modelo.teste@allure.com',
                    'first_name': 'Modelo',
                    'last_name': 'Teste',
                    'is_active': True
                }
            )
            
            if created:
                user.set_password('teste123')
                user.save()
                self.stdout.write(
                    self.style.SUCCESS('✅ Usuário de teste criado')
                )
            else:
                self.stdout.write(
                    self.style.WARNING('🔄 Usuário de teste já existe')
                )
            
            # Criar modelo de teste
            modelo, created = Modelo.objects.get_or_create(
                user=user,
                defaults={
                    'nome': 'Modelo',
                    'sobrenome': 'Teste',
                    'cpf': '12345678901',
                    'data_nascimento': '1990-01-01',
                    'nome_exibicao': 'Modelo Teste',
                    'idade': 30,
                    'estado': 'SP',
                    'cidade': 'São Paulo',
                    'etnia': 'branca',
                    'categoria_servico': 'JOB',
                    'tipo_ensaio': 'solo',
                    'local_atendimento': 'local_proprio',
                    'status': 'cadastro_inicial'
                }
            )
            
            if created:
                self.stdout.write(
                    self.style.SUCCESS('✅ Modelo de teste criada')
                )
            else:
                self.stdout.write(
                    self.style.WARNING('🔄 Modelo de teste já existe')
                )
            
            self.stdout.write(
                self.style.SUCCESS(f'📊 Status da modelo: {modelo.get_status_display()}')
            )
            
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao criar modelo: {str(e)}')
            )
    
    def testar_selecao_plano(self):
        """Testa a seleção de plano e criação de pagamento"""
        self.stdout.write('\n💳 Teste 2: Testando seleção de plano...')
        
        try:
            # Buscar modelo de teste
            try:
                modelo = Modelo.objects.get(user__username='modelo_teste')
            except Modelo.DoesNotExist:
                self.stdout.write(
                    self.style.ERROR('❌ Modelo de teste não encontrada. Execute --criar-modelo-teste primeiro.')
                )
                return
            
            # Buscar plano básico
            try:
                plano = PlanoAnuncio.objects.get(nome='Básico', ativo=True)
            except PlanoAnuncio.DoesNotExist:
                self.stdout.write(
                    self.style.ERROR('❌ Plano Básico não encontrado')
                )
                return
            
            self.stdout.write(
                self.style.SUCCESS(f'✅ Plano selecionado: {plano.nome} - R$ {plano.preco}')
            )
            
            # Testar criação de pagamento
            transacao = PlanoAnuncioService.criar_pagamento_plano(
                modelo=modelo,
                plano_id=plano.id,
                email_pagador=modelo.user.email,
                nome_pagador=f"{modelo.user.first_name} {modelo.user.last_name}"
            )
            
            if transacao:
                self.stdout.write(
                    self.style.SUCCESS(f'✅ Transação criada: {transacao.id}')
                )
                self.stdout.write(f'   • Tipo: {transacao.tipo}')
                self.stdout.write(f'   • Valor: R$ {transacao.valor}')
                self.stdout.write(f'   • Status: {transacao.status}')
                
                # Verificar se o plano foi criado
                try:
                    plano_pagamento = PlanoAnuncioPagamento.objects.get(pagamento=transacao)
                    self.stdout.write(
                        self.style.SUCCESS(f'✅ Plano de pagamento criado: {plano_pagamento.plano.nome}')
                    )
                except PlanoAnuncioPagamento.DoesNotExist:
                    self.stdout.write(
                        self.style.ERROR('❌ Plano de pagamento não foi criado')
                    )
                
                # Testar criação de preferência no Mercado Pago
                try:
                    preference = mercado_pago_service.criar_preferencia_pagamento(transacao)
                    if preference:
                        self.stdout.write(
                            self.style.SUCCESS(f'✅ Preferência MP criada: {preference.get("id")}')
                        )
                        self.stdout.write(f'   • URL de checkout: {preference.get("init_point")}')
                    else:
                        self.stdout.write(
                            self.style.WARNING('⚠️ Preferência MP não foi criada (pode ser normal em desenvolvimento)')
                        )
                except Exception as e:
                    self.stdout.write(
                        self.style.WARNING(f'⚠️ Erro ao criar preferência MP: {str(e)} (pode ser normal em desenvolvimento)')
                    )
                
            else:
                self.stdout.write(
                    self.style.ERROR('❌ Falha ao criar transação')
                )
                
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao testar seleção de plano: {str(e)}')
            )
    
    def simular_pagamento_aprovado(self):
        """Simula pagamento aprovado e testa ativação do plano"""
        self.stdout.write('\n✅ Teste 3: Simulando pagamento aprovado...')
        
        try:
            # Buscar transação pendente
            try:
                transacao = TransacaoPagamento.objects.filter(
                    tipo='plano_anuncio',
                    status='pendente'
                ).first()
                
                if not transacao:
                    self.stdout.write(
                        self.style.WARNING('⚠️ Nenhuma transação pendente encontrada. Execute --testar-selecao-plano primeiro.')
                    )
                    return
                
            except TransacaoPagamento.DoesNotExist:
                self.stdout.write(
                    self.style.ERROR('❌ Transação não encontrada')
                )
                return
            
            self.stdout.write(
                self.style.SUCCESS(f'✅ Transação encontrada: {transacao.id}')
            )
            
            # Simular aprovação
            transacao.status = 'aprovado'
            transacao.data_aprovacao = timezone.now()
            transacao.save()
            
            self.stdout.write(
                self.style.SUCCESS('✅ Status da transação alterado para aprovado')
            )
            
            # Testar ativação do plano
            sucesso = PlanoAnuncioService.ativar_plano_anuncio(transacao)
            
            if sucesso:
                self.stdout.write(
                    self.style.SUCCESS('✅ Plano ativado com sucesso!')
                )
                
                # Verificar status da modelo
                modelo = transacao.modelo
                modelo.refresh_from_db()
                
                self.stdout.write(
                    self.style.SUCCESS(f'📊 Status da modelo: {modelo.get_status_display()}')
                )
                self.stdout.write(f'   • Pagamento aprovado: {"Sim" if modelo.pagamento_aprovado else "Não"}')
                self.stdout.write(f'   • Plano ativo: {modelo.plano_ativo.nome if modelo.plano_ativo else "Nenhum"}')
                self.stdout.write(f'   • Premium: {"Sim" if modelo.is_boost_ativo else "Não"}')
                self.stdout.write(f'   • Boost ativo: {"Sim" if modelo.is_boost_ativo else "Não"}')
                
                # Verificar plano de pagamento
                try:
                    plano_pagamento = PlanoAnuncioPagamento.objects.get(pagamento=transacao)
                    self.stdout.write(
                        self.style.SUCCESS(f'📅 Plano ativo até: {plano_pagamento.data_fim}')
                    )
                except PlanoAnuncioPagamento.DoesNotExist:
                    self.stdout.write(
                        self.style.WARNING('⚠️ Plano de pagamento não encontrado')
                    )
                
            else:
                self.stdout.write(
                    self.style.ERROR('❌ Falha ao ativar plano')
                )
                
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao simular pagamento: {str(e)}')
            )
        
        self.stdout.write(
            self.style.SUCCESS('\n🎉 Testes do fluxo de pagamento concluídos!')
        )
