from django import forms
from django.contrib.auth.models import User
from django.contrib.auth.forms import UserCreationForm
from models.models import Modelo, FotoModelo, Servico


class ModeloRegistrationForm(UserCreationForm):
    """Formulário de cadastro de modelo"""
    
    # Campos do usuário
    email = forms.EmailField(required=True)
    
    # Campos do modelo
    nome_exibicao = forms.CharField(max_length=100, required=True)
    idade = forms.IntegerField(min_value=18, max_value=99, required=True)
    estado = forms.CharField(max_length=2, required=True)
    cidade = forms.CharField(max_length=100, required=True)
    etnia = forms.ChoiceField(
        choices=Modelo.ETNIA_CHOICES,
        required=True
    )
    categoria_servico = forms.MultipleChoiceField(
        choices=Modelo.CATEGORIA_SERVICO_CHOICES,
        widget=forms.CheckboxSelectMultiple,
        required=True,
        label='Tipos de Serviço'
    )
    tipo_ensaio = forms.ChoiceField(
        choices=Modelo.TIPO_ENSAIO_CHOICES,
        required=True
    )
    local_atendimento = forms.MultipleChoiceField(
        choices=Modelo.LOCAL_ATENDIMENTO_CHOICES,
        widget=forms.CheckboxSelectMultiple,
        required=True,
        label='Locais de Atendimento'
    )
    
    # Campos opcionais
    altura = forms.IntegerField(required=False, min_value=140, max_value=220)
    peso = forms.IntegerField(required=False, min_value=40, max_value=150)
    sobre_mim = forms.CharField(widget=forms.Textarea, required=False)
    
    # Contatos
    telefone = forms.CharField(max_length=20, required=False)
    whatsapp = forms.CharField(max_length=20, required=False)
    email_contato = forms.EmailField(required=False)
    
    # Arquivos
    foto_perfil = forms.ImageField(required=True)
    documento_rg = forms.ImageField(required=True)
    documento_selfie = forms.ImageField(required=True)
    
    # Termos
    aceito_termos = forms.BooleanField(required=True)
    
    class Meta:
        model = User
        fields = ('username', 'email', 'password1', 'password2')
    
    def clean_email(self):
        email = self.cleaned_data.get('email')
        if User.objects.filter(email=email).exists():
            raise forms.ValidationError("Este e-mail já está em uso.")
        return email
    
    def clean_whatsapp(self):
        whatsapp = self.cleaned_data.get('whatsapp')
        if whatsapp:
            # Remove formatação
            whatsapp = ''.join(filter(str.isdigit, whatsapp))
            if len(whatsapp) < 10:
                raise forms.ValidationError("WhatsApp deve ter pelo menos 10 dígitos.")
        return whatsapp
    
    def save(self, commit=True):
        user = super().save(commit=False)
        user.email = self.cleaned_data['email']
        
        if commit:
            user.save()
            
            # Criar o modelo
            modelo = Modelo.objects.create(
                user=user,
                nome_exibicao=self.cleaned_data['nome_exibicao'],
                idade=self.cleaned_data['idade'],
                estado=self.cleaned_data['estado'].upper(),
                cidade=self.cleaned_data['cidade'],
                etnia=self.cleaned_data['etnia'],
                categoria_servico=self.cleaned_data['categoria_servico'],
                tipo_ensaio=self.cleaned_data['tipo_ensaio'],
                local_atendimento=self.cleaned_data['local_atendimento'],
                altura=self.cleaned_data.get('altura'),
                peso=self.cleaned_data.get('peso'),
                sobre_mim=self.cleaned_data.get('sobre_mim', ''),
                telefone=self.cleaned_data.get('telefone', ''),
                whatsapp=self.cleaned_data.get('whatsapp', ''),
                email_contato=self.cleaned_data.get('email_contato', ''),
                foto_perfil=self.cleaned_data['foto_perfil'],
                documento_rg=self.cleaned_data['documento_rg'],
                documento_selfie=self.cleaned_data['documento_selfie'],
                status='pendente'  # Aguardando aprovação
            )
            
        return user


class FiltroModelosForm(forms.Form):
    """Formulário de filtros para listagem de modelos"""
    
    categoria_servico = forms.MultipleChoiceField(
        choices=Modelo.CATEGORIA_SERVICO_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple,
        label='Tipos de Serviço'
    )
    
    tipo = forms.MultipleChoiceField(
        choices=Modelo.TIPO_ENSAIO_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple
    )
    
    etnia = forms.MultipleChoiceField(
        choices=Modelo.ETNIA_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple
    )
    
    local_atendimento = forms.MultipleChoiceField(
        choices=Modelo.LOCAL_ATENDIMENTO_CHOICES,
        required=False,
        widget=forms.CheckboxSelectMultiple,
        label='Locais de Atendimento'
    )
    
    estado = forms.CharField(max_length=2, required=False)
    cidade = forms.CharField(max_length=100, required=False)
    
    idade_min = forms.IntegerField(min_value=18, max_value=99, required=False)
    idade_max = forms.IntegerField(min_value=18, max_value=99, required=False)
    
    ordenacao = forms.ChoiceField(
        choices=[
            ('recentes', 'Mais Recentes'),
            ('populares', 'Mais Vistas (Total)'),
            ('populares_mes', 'Mais Vistas (Mês)'),
        ],
        required=False,
        initial='recentes'
    )
    
    def clean(self):
        cleaned_data = super().clean()
        idade_min = cleaned_data.get('idade_min')
        idade_max = cleaned_data.get('idade_max')
        
        if idade_min and idade_max and idade_min > idade_max:
            raise forms.ValidationError("Idade mínima não pode ser maior que a máxima.")
        
        return cleaned_data


class ServicoForm(forms.ModelForm):
    """Formulário para serviços da modelo"""
    
    class Meta:
        model = Servico
        fields = ['descricao', 'preco', 'duracao']
        widgets = {
            'descricao': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Ex: Acompanhante para eventos'
            }),
            'preco': forms.NumberInput(attrs={
                'class': 'form-control',
                'placeholder': '200',
                'step': '0.01'
            }),
            'duracao': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '1 hora'
            })
        }


class FotoGaleriaForm(forms.ModelForm):
    """Formulário para upload de fotos da galeria"""
    
    class Meta:
        model = FotoModelo
        fields = ['imagem', 'categoria', 'autorizar_redes_sociais']
        widgets = {
            'categoria': forms.Select(attrs={'class': 'form-select'}),
            'imagem': forms.FileInput(attrs={
                'class': 'form-control',
                'accept': 'image/*'
            }),
            'autorizar_redes_sociais': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
        labels = {
            'autorizar_redes_sociais': 'Autorizar uso em redes sociais'
        }
        help_texts = {
            'autorizar_redes_sociais': 'Marque se autoriza o uso desta foto em redes sociais da plataforma'
        }


class BoostAnuncioForm(forms.Form):
    """Formulário para boost de anúncios"""
    
    DURACAO_CHOICES = [
        (1, '1 dia - R$ 10,00'),
        (3, '3 dias - R$ 25,00'),
        (7, '7 dias - R$ 50,00'),
        (15, '15 dias - R$ 90,00'),
        (30, '30 dias - R$ 150,00'),
    ]
    
    duracao = forms.ChoiceField(
        choices=DURACAO_CHOICES,
        required=True,
        widget=forms.RadioSelect
    )
    
    def get_preco(self):
        """Retorna o preço baseado na duração escolhida"""
        precos = {
            1: 10.00,
            3: 25.00,
            7: 50.00,
            15: 90.00,
            30: 150.00,
        }
        duracao = int(self.cleaned_data.get('duracao', 1))
        return precos.get(duracao, 10.00)


class ContatoForm(forms.Form):
    """Formulário de contato"""
    
    nome = forms.CharField(max_length=100, required=True)
    email = forms.EmailField(required=True)
    assunto = forms.CharField(max_length=200, required=True)
    mensagem = forms.CharField(widget=forms.Textarea, required=True)
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for field in self.fields:
            self.fields[field].widget.attrs.update({'class': 'form-control'})


class BuscaForm(forms.Form):
    """Formulário de busca geral"""
    
    q = forms.CharField(
        max_length=200,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Buscar modelos...'
        })
    )
    
    localidade = forms.CharField(
        max_length=100,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Cidade ou Estado...'
        })
    )

