"""
Sistema de gatilhos para eventos do sistema
"""
from django.db.models.signals import post_save, pre_save
from django.dispatch import receiver
from django.utils import timezone
from datetime import timedelta
import logging

from .services import (
    trigger_modelo_cadastrado,
    trigger_modelo_aprovado,
    trigger_pagamento_aprovado,
    trigger_plano_ativado
)
from .models import EmailTrigger
from models.models import Modelo
from payments.models import TransacaoPagamento

logger = logging.getLogger(__name__)


@receiver(post_save, sender=Modelo)
def modelo_saved(sender, instance, created, **kwargs):
    """Gatilho quando uma modelo é salva"""
    try:
        if created:
            # Nova modelo cadastrada
            logger.info(f"Modelo {instance.id} cadastrada - disparando gatilho")
            trigger_modelo_cadastrado(instance)
        
        else:
            # Modelo atualizada - verificar se foi aprovada
            if hasattr(instance, '_previous_status'):
                if (instance._previous_status != 'aprovado' and 
                    instance.status == 'aprovado'):
                    logger.info(f"Modelo {instance.id} aprovada - disparando gatilho")
                    trigger_modelo_aprovado(instance)
    except Exception as e:
        logger.error(f"Erro no gatilho de modelo: {str(e)}")


@receiver(pre_save, sender=Modelo)
def modelo_pre_save(sender, instance, **kwargs):
    """Captura o status anterior da modelo"""
    try:
        if instance.pk:
            old_instance = Modelo.objects.get(pk=instance.pk)
            instance._previous_status = old_instance.status
        else:
            instance._previous_status = None
    except Modelo.DoesNotExist:
        instance._previous_status = None
    except Exception as e:
        logger.error(f"Erro ao capturar status anterior da modelo: {str(e)}")
        instance._previous_status = None


@receiver(post_save, sender=TransacaoPagamento)
def pagamento_saved(sender, instance, created, **kwargs):
    """Gatilho quando um pagamento é salvo"""
    try:
        if not created:
            # Pagamento atualizado - verificar se foi aprovado
            if hasattr(instance, '_previous_status'):
                if (instance._previous_status != 'aprovado' and 
                    instance.status == 'aprovado'):
                    logger.info(f"Pagamento {instance.id} aprovado - disparando gatilho")
                    trigger_pagamento_aprovado(instance, instance.modelo)
    except Exception as e:
        logger.error(f"Erro no gatilho de pagamento: {str(e)}")


@receiver(pre_save, sender=TransacaoPagamento)
def pagamento_pre_save(sender, instance, **kwargs):
    """Captura o status anterior do pagamento"""
    try:
        if instance.pk:
            old_instance = TransacaoPagamento.objects.get(pk=instance.pk)
            instance._previous_status = old_instance.status
        else:
            instance._previous_status = None
    except TransacaoPagamento.DoesNotExist:
        instance._previous_status = None
    except Exception as e:
        logger.error(f"Erro ao capturar status anterior do pagamento: {str(e)}")
        instance._previous_status = None


# Gatilhos manuais para eventos específicos
def trigger_plano_expirando():
    """Gatilho para planos que expiram em breve"""
    from .services import TriggerService
    
    try:
        # Buscar planos que expiram em 7 dias
        data_limite_7_dias = timezone.now() + timedelta(days=7)
        data_limite_3_dias = timezone.now() + timedelta(days=3)
        
        modelos_7_dias = Modelo.objects.filter(
            plano_data_fim__lte=data_limite_7_dias,
            plano_data_fim__gt=data_limite_3_dias,
            status='ativo'
        )
        
        modelos_3_dias = Modelo.objects.filter(
            plano_data_fim__lte=data_limite_3_dias,
            plano_data_fim__gt=timezone.now(),
            status='ativo'
        )
        
        # Processar modelos que expiram em 7 dias
        for modelo in modelos_7_dias:
            contexto = {
                'nome': f"{modelo.nome} {modelo.sobrenome}".strip(),
                'email': modelo.user.email,
                'plano_nome': modelo.plano_ativo.nome if modelo.plano_ativo else 'Plano Ativo',
                'data_expiracao': modelo.plano_data_fim.strftime('%d/%m/%Y'),
                'dias_restantes': 7
            }
            TriggerService.process_trigger('plano_expira_em_7_dias', contexto, modelo=modelo)
        
        # Processar modelos que expiram em 3 dias
        for modelo in modelos_3_dias:
            contexto = {
                'nome': f"{modelo.nome} {modelo.sobrenome}".strip(),
                'email': modelo.user.email,
                'plano_nome': modelo.plano_ativo.nome if modelo.plano_ativo else 'Plano Ativo',
                'data_expiracao': modelo.plano_data_fim.strftime('%d/%m/%Y'),
                'dias_restantes': 3
            }
            TriggerService.process_trigger('plano_expira_em_3_dias', contexto, modelo=modelo)
        
        logger.info(f"Gatilho de expiração processado: {modelos_7_dias.count()} em 7 dias, {modelos_3_dias.count()} em 3 dias")
        
    except Exception as e:
        logger.error(f"Erro no gatilho de expiração de planos: {str(e)}")


def trigger_planos_expirados():
    """Gatilho para planos expirados"""
    from .services import TriggerService
    
    try:
        modelos_expiradas = Modelo.objects.filter(
            plano_data_fim__lt=timezone.now(),
            status='ativo'
        )
        
        for modelo in modelos_expiradas:
            contexto = {
                'nome': f"{modelo.nome} {modelo.sobrenome}".strip(),
                'email': modelo.user.email,
                'plano_nome': modelo.plano_ativo.nome if modelo.plano_ativo else 'Plano Ativo',
                'data_expiracao': modelo.plano_data_fim.strftime('%d/%m/%Y'),
            }
            TriggerService.process_trigger('plano_expirado', contexto, modelo=modelo)
        
        logger.info(f"Gatilho de planos expirados processado: {modelos_expiradas.count()} modelos")
        
    except Exception as e:
        logger.error(f"Erro no gatilho de planos expirados: {str(e)}")


def trigger_contato_recebido(nome, email, mensagem, assunto=""):
    """Gatilho para contato recebido"""
    from .services import TriggerService
    
    try:
        contexto = {
            'nome_cliente': nome,
            'email_cliente': email,
            'mensagem': mensagem,
            'assunto': assunto,
            'data_contato': timezone.now().strftime('%d/%m/%Y %H:%M'),
        }
        TriggerService.process_trigger('contato_recebido', contexto)
        logger.info(f"Gatilho de contato processado: {email}")
        
    except Exception as e:
        logger.error(f"Erro no gatilho de contato: {str(e)}")


def trigger_recuperacao_senha(user, token):
    """Gatilho para recuperação de senha"""
    from .services import TriggerService
    
    try:
        contexto = {
            'nome': f"{user.first_name} {user.last_name}".strip(),
            'email': user.email,
            'token': token,
            'link_recuperacao': f"{getattr(settings, 'SITE_URL', 'https://allure.com.br')}/redefinir-senha/{token}",
        }
        TriggerService.process_trigger('senha_redefinida', contexto, user=user)
        logger.info(f"Gatilho de recuperação de senha processado: {user.email}")
        
    except Exception as e:
        logger.error(f"Erro no gatilho de recuperação de senha: {str(e)}")


# Função para inicializar gatilhos padrão
def criar_gatilhos_padrao():
    """Cria gatilhos padrão do sistema"""
    from .models import EmailTemplate
    
    gatilhos_padrao = [
        {
            'nome': 'Cadastro de Modelo',
            'evento': 'modelo_cadastrado',
            'template_tipo': 'cadastro_sucesso',
            'delay_minutos': 0,
        },
        {
            'nome': 'Modelo Aprovada',
            'evento': 'modelo_aprovado',
            'template_tipo': 'perfil_aprovado',
            'delay_minutos': 0,
        },
        {
            'nome': 'Pagamento Aprovado',
            'evento': 'pagamento_aprovado',
            'template_tipo': 'pagamento_aprovado',
            'delay_minutos': 0,
        },
        {
            'nome': 'Plano Ativado',
            'evento': 'plano_ativado',
            'template_tipo': 'plano_ativado',
            'delay_minutos': 0,
        },
        {
            'nome': 'Plano Expira em 7 Dias',
            'evento': 'plano_expira_em_7_dias',
            'template_tipo': 'plano_expirando',
            'delay_minutos': 0,
        },
        {
            'nome': 'Plano Expira em 3 Dias',
            'evento': 'plano_expira_em_3_dias',
            'template_tipo': 'plano_expirando',
            'delay_minutos': 0,
        },
        {
            'nome': 'Plano Expirado',
            'evento': 'plano_expirado',
            'template_tipo': 'plano_expirado',
            'delay_minutos': 0,
        },
    ]
    
    for gatilho_data in gatilhos_padrao:
        try:
            # Buscar template
            template = EmailTemplate.objects.filter(
                tipo=gatilho_data['template_tipo'],
                ativo=True
            ).first()
            
            if template:
                # Criar ou atualizar gatilho
                gatilho, created = EmailTrigger.objects.get_or_create(
                    evento=gatilho_data['evento'],
                    defaults={
                        'nome': gatilho_data['nome'],
                        'template': template,
                        'delay_minutos': gatilho_data['delay_minutos'],
                        'ativo': True
                    }
                )
                
                if created:
                    logger.info(f"Gatilho criado: {gatilho.nome}")
                else:
                    logger.info(f"Gatilho já existe: {gatilho.nome}")
            else:
                logger.warning(f"Template não encontrado para gatilho: {gatilho_data['template_tipo']}")
                
        except Exception as e:
            logger.error(f"Erro ao criar gatilho {gatilho_data['nome']}: {str(e)}")
