"""
Serviços para envio de emails e processamento de gatilhos
"""
from django.core.mail import EmailMultiAlternatives
from django.template.loader import render_to_string
from django.template import Template, Context
from django.conf import settings
from django.utils import timezone
from django.db import transaction
import logging
import json
from typing import Dict, Any, Optional, List
from datetime import timedelta

from .models import EmailTemplate, EmailLog, EmailQueue, EmailTrigger
from models.models import Modelo

logger = logging.getLogger(__name__)


class EmailService:
    """Serviço principal para envio de emails"""
    
    @staticmethod
    def render_template(template: EmailTemplate, contexto: Dict[str, Any]) -> tuple:
        """
        Renderiza um template de email com o contexto fornecido
        
        Args:
            template: Template de email
            contexto: Dicionário com variáveis para o template
            
        Returns:
            Tupla (assunto, conteudo_html, conteudo_texto)
        """
        try:
            # Renderizar assunto
            assunto_template = Template(template.assunto)
            assunto = assunto_template.render(Context(contexto))
            
            # Renderizar conteúdo HTML
            conteudo_html_template = Template(template.conteudo_html)
            conteudo_html = conteudo_html_template.render(Context(contexto))
            
            # Renderizar conteúdo texto (se existir)
            conteudo_texto = ""
            if template.conteudo_texto:
                conteudo_texto_template = Template(template.conteudo_texto)
                conteudo_texto = conteudo_texto_template.render(Context(contexto))
            
            return assunto, conteudo_html, conteudo_texto
            
        except Exception as e:
            logger.error(f"Erro ao renderizar template {template.id}: {str(e)}")
            raise
    
    @staticmethod
    def send_email(
        template: EmailTemplate,
        destinatario_email: str,
        contexto: Dict[str, Any],
        destinatario_nome: str = "",
        user: Optional[Any] = None,
        modelo: Optional[Modelo] = None,
        prioridade: int = 5
    ) -> EmailLog:
        """
        Envia um email usando um template
        
        Args:
            template: Template de email
            destinatario_email: Email do destinatário
            contexto: Contexto para renderizar o template
            destinatario_nome: Nome do destinatário
            user: Usuário relacionado (opcional)
            modelo: Modelo relacionada (opcional)
            prioridade: Prioridade do email (1=alta, 5=normal, 10=baixa)
            
        Returns:
            EmailLog criado
        """
        try:
            # Renderizar template
            assunto, conteudo_html, conteudo_texto = EmailService.render_template(
                template, contexto
            )
            
            # Criar log de email
            email_log = EmailLog.objects.create(
                template=template,
                destinatario_email=destinatario_email,
                destinatario_nome=destinatario_nome,
                assunto=assunto,
                contexto=contexto,
                user=user,
                modelo=modelo
            )
            
            # Enviar email
            if settings.EMAIL_BACKEND == 'django.core.mail.backends.console.EmailBackend':
                # Modo desenvolvimento - apenas log
                logger.info(f"EMAIL (DEV): {assunto} -> {destinatario_email}")
                email_log.status = 'enviado'
                email_log.save()
            else:
                # Modo produção - envio real
                try:
                    msg = EmailMultiAlternatives(
                        subject=assunto,
                        body=conteudo_texto or conteudo_html,
                        from_email=settings.DEFAULT_FROM_EMAIL,
                        to=[destinatario_email]
                    )
                    
                    if conteudo_html:
                        msg.attach_alternative(conteudo_html, "text/html")
                    
                    msg.send()
                    email_log.status = 'enviado'
                    email_log.data_entrega = timezone.now()
                    email_log.save()
                    
                except Exception as e:
                    email_log.status = 'falhou'
                    email_log.erro_mensagem = str(e)
                    email_log.save()
                    logger.error(f"Erro ao enviar email: {str(e)}")
            
            return email_log
            
        except Exception as e:
            logger.error(f"Erro no serviço de email: {str(e)}")
            raise
    
    @staticmethod
    def queue_email(
        template: EmailTemplate,
        destinatario_email: str,
        contexto: Dict[str, Any],
        destinatario_nome: str = "",
        user: Optional[Any] = None,
        modelo: Optional[Modelo] = None,
        prioridade: int = 5,
        delay_minutos: int = 0
    ) -> EmailQueue:
        """
        Adiciona um email à fila para processamento assíncrono
        
        Args:
            template: Template de email
            destinatario_email: Email do destinatário
            contexto: Contexto para renderizar o template
            destinatario_nome: Nome do destinatário
            user: Usuário relacionado (opcional)
            modelo: Modelo relacionada (opcional)
            prioridade: Prioridade do email (1=alta, 5=normal, 10=baixa)
            delay_minutos: Delay em minutos antes do envio
            
        Returns:
            EmailQueue criado
        """
        data_agendamento = timezone.now() + timedelta(minutes=delay_minutos)
        
        queue_item = EmailQueue.objects.create(
            template=template,
            destinatario_email=destinatario_email,
            destinatario_nome=destinatario_nome,
            contexto=contexto,
            prioridade=prioridade,
            data_agendamento=data_agendamento,
            user=user,
            modelo=modelo
        )
        
        return queue_item


class TriggerService:
    """Serviço para processamento de gatilhos automáticos"""
    
    @staticmethod
    def process_trigger(
        evento: str,
        contexto: Dict[str, Any],
        user: Optional[Any] = None,
        modelo: Optional[Modelo] = None
    ) -> List[EmailQueue]:
        """
        Processa um gatilho de evento e envia emails correspondentes
        
        Args:
            evento: Nome do evento
            contexto: Contexto do evento
            user: Usuário relacionado (opcional)
            modelo: Modelo relacionada (opcional)
            
        Returns:
            Lista de EmailQueue criados
        """
        try:
            # Buscar gatilhos ativos para o evento
            triggers = EmailTrigger.objects.filter(
                evento=evento,
                ativo=True
            ).select_related('template')
            
            queue_items = []
            
            for trigger in triggers:
                # Verificar condições adicionais
                if not TriggerService._verificar_condicoes(trigger, contexto):
                    continue
                
                # Preparar contexto do email
                email_contexto = TriggerService._preparar_contexto(trigger, contexto)
                
                # Determinar destinatário
                destinatario_email, destinatario_nome = TriggerService._determinar_destinatario(
                    trigger, user, modelo, contexto
                )
                
                if not destinatario_email:
                    logger.warning(f"Gatilho {trigger.id}: destinatário não encontrado")
                    continue
                
                # Adicionar à fila
                queue_item = EmailService.queue_email(
                    template=trigger.template,
                    destinatario_email=destinatario_email,
                    contexto=email_contexto,
                    destinatario_nome=destinatario_nome,
                    user=user,
                    modelo=modelo,
                    prioridade=3,  # Prioridade média para gatilhos
                    delay_minutos=trigger.delay_minutos
                )
                
                queue_items.append(queue_item)
                logger.info(f"Gatilho {trigger.id} processado: email agendado para {destinatario_email}")
            
            return queue_items
            
        except Exception as e:
            logger.error(f"Erro ao processar gatilho {evento}: {str(e)}")
            return []
    
    @staticmethod
    def _verificar_condicoes(trigger: EmailTrigger, contexto: Dict[str, Any]) -> bool:
        """Verifica se as condições do gatilho são atendidas"""
        if not trigger.condicoes:
            return True
        
        try:
            for campo, valor_esperado in trigger.condicoes.items():
                valor_atual = contexto.get(campo)
                if valor_atual != valor_esperado:
                    return False
            return True
        except Exception as e:
            logger.error(f"Erro ao verificar condições do gatilho {trigger.id}: {str(e)}")
            return False
    
    @staticmethod
    def _preparar_contexto(trigger: EmailTrigger, contexto: Dict[str, Any]) -> Dict[str, Any]:
        """Prepara o contexto para o template de email"""
        # Adicionar informações básicas do site
        email_contexto = {
            'site_url': getattr(settings, 'SITE_URL', 'https://allure.com.br'),
            'site_nome': 'Allure',
            'data_atual': timezone.now().strftime('%d/%m/%Y'),
            'hora_atual': timezone.now().strftime('%H:%M'),
        }
        
        # Adicionar contexto do evento
        email_contexto.update(contexto)
        
        return email_contexto
    
    @staticmethod
    def _determinar_destinatario(
        trigger: EmailTrigger,
        user: Optional[Any],
        modelo: Optional[Modelo],
        contexto: Dict[str, Any]
    ) -> tuple:
        """Determina o destinatário do email"""
        # Prioridade: contexto > modelo > user
        if 'email' in contexto:
            return contexto['email'], contexto.get('nome', '')
        
        if modelo and modelo.user and modelo.user.email:
            nome = f"{modelo.nome} {modelo.sobrenome}".strip()
            return modelo.user.email, nome
        
        if user and user.email:
            nome = f"{user.first_name} {user.last_name}".strip()
            return user.email, nome
        
        return None, None


class QueueProcessor:
    """Processador da fila de emails"""
    
    @staticmethod
    def process_queue(batch_size: int = 10) -> Dict[str, int]:
        """
        Processa itens da fila de emails
        
        Args:
            batch_size: Número máximo de emails para processar
            
        Returns:
            Dicionário com estatísticas do processamento
        """
        stats = {
            'processados': 0,
            'enviados': 0,
            'falharam': 0,
            'erros': []
        }
        
        try:
            # Buscar itens pendentes
            queue_items = EmailQueue.objects.filter(
                status='pendente',
                data_agendamento__lte=timezone.now()
            ).select_related('template')[:batch_size]
            
            for item in queue_items:
                try:
                    # Marcar como processando
                    item.status = 'processando'
                    item.data_processamento = timezone.now()
                    item.save()
                    
                    # Enviar email
                    email_log = EmailService.send_email(
                        template=item.template,
                        destinatario_email=item.destinatario_email,
                        contexto=item.contexto,
                        destinatario_nome=item.destinatario_nome,
                        user=item.user,
                        modelo=item.modelo
                    )
                    
                    # Atualizar status
                    if email_log.status == 'enviado':
                        item.status = 'enviado'
                        stats['enviados'] += 1
                    else:
                        item.status = 'falhou'
                        item.erro_mensagem = email_log.erro_mensagem
                        stats['falharam'] += 1
                    
                    item.save()
                    stats['processados'] += 1
                    
                except Exception as e:
                    # Marcar como falhou
                    item.status = 'falhou'
                    item.erro_mensagem = str(e)
                    item.tentativas += 1
                    item.save()
                    
                    stats['falharam'] += 1
                    stats['erros'].append(str(e))
                    logger.error(f"Erro ao processar item da fila {item.id}: {str(e)}")
            
            return stats
            
        except Exception as e:
            logger.error(f"Erro no processador da fila: {str(e)}")
            stats['erros'].append(str(e))
            return stats
    
    @staticmethod
    def retry_failed_emails(max_retries: int = 3) -> int:
        """
        Tenta reenviar emails que falharam
        
        Args:
            max_retries: Número máximo de tentativas
            
        Returns:
            Número de emails reprocessados
        """
        try:
            failed_items = EmailQueue.objects.filter(
                status='falhou',
                tentativas__lt=max_retries
            )
            
            reprocessados = 0
            for item in failed_items:
                item.status = 'pendente'
                item.tentativas += 1
                item.data_agendamento = timezone.now()
                item.save()
                reprocessados += 1
            
            return reprocessados
            
        except Exception as e:
            logger.error(f"Erro ao reprocessar emails falhados: {str(e)}")
            return 0


# Funções de conveniência para eventos específicos
def trigger_modelo_cadastrado(modelo: Modelo):
    """Gatilho para modelo cadastrada"""
    contexto = {
        'nome': f"{modelo.nome} {modelo.sobrenome}".strip(),
        'email': modelo.user.email,
        'modelo_id': modelo.id,
        'data_cadastro': modelo.data_cadastro.strftime('%d/%m/%Y') if modelo.data_cadastro else '',
    }
    return TriggerService.process_trigger('modelo_cadastrado', contexto, modelo=modelo)


def trigger_modelo_aprovado(modelo: Modelo):
    """Gatilho para modelo aprovada"""
    contexto = {
        'nome': f"{modelo.nome} {modelo.sobrenome}".strip(),
        'email': modelo.user.email,
        'modelo_id': modelo.id,
        'data_aprovacao': timezone.now().strftime('%d/%m/%Y'),
    }
    return TriggerService.process_trigger('modelo_aprovado', contexto, modelo=modelo)


def trigger_pagamento_aprovado(transacao, modelo: Modelo = None):
    """Gatilho para pagamento aprovado"""
    contexto = {
        'nome': transacao.nome_pagador,
        'email': transacao.email_pagador,
        'valor': f"R$ {transacao.valor:.2f}",
        'descricao': transacao.descricao,
        'data_pagamento': transacao.data_aprovacao.strftime('%d/%m/%Y') if transacao.data_aprovacao else '',
        'tipo_pagamento': transacao.tipo,
    }
    return TriggerService.process_trigger('pagamento_aprovado', contexto, modelo=modelo)


def trigger_plano_ativado(modelo: Modelo, plano):
    """Gatilho para plano ativado"""
    contexto = {
        'nome': f"{modelo.nome} {modelo.sobrenome}".strip(),
        'email': modelo.user.email,
        'plano_nome': plano.nome,
        'plano_tipo': plano.tipo,
        'data_inicio': modelo.plano_data_inicio.strftime('%d/%m/%Y') if modelo.plano_data_inicio else '',
        'data_fim': modelo.plano_data_fim.strftime('%d/%m/%Y') if modelo.plano_data_fim else '',
    }
    return TriggerService.process_trigger('plano_ativado', contexto, modelo=modelo)
