"""
Modelos para o sistema de comunicação e notificações por email
"""
from django.db import models
from django.contrib.auth.models import User
from django.utils import timezone
from django.core.validators import MinLengthValidator
import json


class EmailTemplate(models.Model):
    """Template de email para diferentes tipos de notificação"""
    
    TIPO_CHOICES = [
        ('cadastro_sucesso', 'Cadastro Realizado com Sucesso'),
        ('perfil_aprovado', 'Perfil Aprovado'),
        ('perfil_rejeitado', 'Perfil Rejeitado'),
        ('pagamento_aprovado', 'Pagamento Aprovado'),
        ('pagamento_pendente', 'Pagamento Pendente'),
        ('pagamento_falhou', 'Pagamento Falhou'),
        ('plano_ativado', 'Plano Ativado'),
        ('plano_expirando', 'Plano Expirando em Breve'),
        ('plano_expirado', 'Plano Expirado'),
        ('boost_ativado', 'Boost Ativado'),
        ('boost_expirado', 'Boost Expirado'),
        ('premium_ativado', 'Premium Ativado'),
        ('premium_expirado', 'Premium Expirado'),
        ('recuperacao_senha', 'Recuperação de Senha'),
        ('contato_cliente', 'Mensagem de Cliente'),
        ('bem_vindo', 'Bem-vindo ao Allure'),
        ('promocao', 'Promoção Especial'),
        ('newsletter', 'Newsletter'),
        ('custom', 'Personalizado'),
    ]
    
    nome = models.CharField(
        max_length=100,
        validators=[MinLengthValidator(3)],
        help_text="Nome identificador do template"
    )
    tipo = models.CharField(
        max_length=50,
        choices=TIPO_CHOICES,
        unique=True,
        help_text="Tipo de notificação"
    )
    assunto = models.CharField(
        max_length=200,
        help_text="Assunto do email (pode usar variáveis como {{nome}})"
    )
    conteudo_html = models.TextField(
        help_text="Conteúdo HTML do email (pode usar variáveis como {{nome}})"
    )
    conteudo_texto = models.TextField(
        blank=True,
        help_text="Conteúdo em texto simples (opcional)"
    )
    ativo = models.BooleanField(
        default=True,
        help_text="Se o template está ativo para uso"
    )
    variaveis_disponiveis = models.JSONField(
        default=list,
        blank=True,
        help_text="Lista de variáveis disponíveis para este template"
    )
    data_criacao = models.DateTimeField(auto_now_add=True)
    data_atualizacao = models.DateTimeField(auto_now=True)
    criado_por = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='templates_criados'
    )
    
    class Meta:
        verbose_name = "Template de Email"
        verbose_name_plural = "Templates de Email"
        ordering = ['tipo', 'nome']
    
    def __str__(self):
        return f"{self.nome} ({self.get_tipo_display()})"
    
    def get_variaveis(self):
        """Retorna as variáveis disponíveis como lista"""
        if isinstance(self.variaveis_disponiveis, list):
            return self.variaveis_disponiveis
        return []
    
    def add_variavel(self, variavel):
        """Adiciona uma variável à lista"""
        variaveis = self.get_variaveis()
        if variavel not in variaveis:
            variaveis.append(variavel)
            self.variaveis_disponiveis = variaveis
            self.save(update_fields=['variaveis_disponiveis'])


class EmailLog(models.Model):
    """Log de emails enviados"""
    
    STATUS_CHOICES = [
        ('pendente', 'Pendente'),
        ('enviado', 'Enviado'),
        ('falhou', 'Falhou'),
        ('entregue', 'Entregue'),
        ('rejeitado', 'Rejeitado'),
    ]
    
    template = models.ForeignKey(
        EmailTemplate,
        on_delete=models.CASCADE,
        related_name='logs'
    )
    destinatario_email = models.EmailField()
    destinatario_nome = models.CharField(max_length=200, blank=True)
    assunto = models.CharField(max_length=200)
    status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='pendente'
    )
    data_envio = models.DateTimeField(auto_now_add=True)
    data_entrega = models.DateTimeField(null=True, blank=True)
    erro_mensagem = models.TextField(blank=True)
    contexto = models.JSONField(
        default=dict,
        blank=True,
        help_text="Contexto usado para renderizar o template"
    )
    user = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='emails_recebidos'
    )
    modelo = models.ForeignKey(
        'models.Modelo',
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='emails_recebidos'
    )
    
    class Meta:
        verbose_name = "Log de Email"
        verbose_name_plural = "Logs de Email"
        ordering = ['-data_envio']
        indexes = [
            models.Index(fields=['destinatario_email', 'data_envio']),
            models.Index(fields=['status', 'data_envio']),
            models.Index(fields=['template', 'data_envio']),
        ]
    
    def __str__(self):
        return f"{self.destinatario_email} - {self.assunto} ({self.get_status_display()})"


class EmailTrigger(models.Model):
    """Configuração de gatilhos automáticos para envio de emails"""
    
    EVENTO_CHOICES = [
        ('modelo_cadastrado', 'Modelo Cadastrado'),
        ('modelo_aprovado', 'Modelo Aprovado'),
        ('modelo_rejeitado', 'Modelo Rejeitado'),
        ('pagamento_criado', 'Pagamento Criado'),
        ('pagamento_aprovado', 'Pagamento Aprovado'),
        ('pagamento_falhou', 'Pagamento Falhou'),
        ('plano_ativado', 'Plano Ativado'),
        ('plano_expira_em_7_dias', 'Plano Expira em 7 Dias'),
        ('plano_expira_em_3_dias', 'Plano Expira em 3 Dias'),
        ('plano_expirado', 'Plano Expirado'),
        ('boost_ativado', 'Boost Ativado'),
        ('boost_expirado', 'Boost Expirado'),
        ('premium_ativado', 'Premium Ativado'),
        ('premium_expirado', 'Premium Expirado'),
        ('senha_redefinida', 'Senha Redefinida'),
        ('contato_recebido', 'Contato Recebido'),
    ]
    
    nome = models.CharField(max_length=100)
    evento = models.CharField(
        max_length=50,
        choices=EVENTO_CHOICES,
        unique=True
    )
    template = models.ForeignKey(
        EmailTemplate,
        on_delete=models.CASCADE,
        related_name='triggers'
    )
    ativo = models.BooleanField(default=True)
    delay_minutos = models.PositiveIntegerField(
        default=0,
        help_text="Delay em minutos antes do envio (0 = imediato)"
    )
    condicoes = models.JSONField(
        default=dict,
        blank=True,
        help_text="Condições adicionais para o envio (ex: status específico)"
    )
    data_criacao = models.DateTimeField(auto_now_add=True)
    data_atualizacao = models.DateTimeField(auto_now=True)
    
    class Meta:
        verbose_name = "Gatilho de Email"
        verbose_name_plural = "Gatilhos de Email"
        ordering = ['evento']
    
    def __str__(self):
        return f"{self.nome} ({self.get_evento_display()})"


class EmailQueue(models.Model):
    """Fila de emails para processamento assíncrono"""
    
    STATUS_CHOICES = [
        ('pendente', 'Pendente'),
        ('processando', 'Processando'),
        ('enviado', 'Enviado'),
        ('falhou', 'Falhou'),
        ('cancelado', 'Cancelado'),
    ]
    
    template = models.ForeignKey(
        EmailTemplate,
        on_delete=models.CASCADE,
        related_name='queue_items'
    )
    destinatario_email = models.EmailField()
    destinatario_nome = models.CharField(max_length=200, blank=True)
    contexto = models.JSONField(
        default=dict,
        help_text="Contexto para renderizar o template"
    )
    status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='pendente'
    )
    prioridade = models.PositiveIntegerField(
        default=5,
        help_text="Prioridade (1=alta, 5=normal, 10=baixa)"
    )
    tentativas = models.PositiveIntegerField(default=0)
    max_tentativas = models.PositiveIntegerField(default=3)
    data_agendamento = models.DateTimeField(
        default=timezone.now,
        help_text="Data/hora para envio"
    )
    data_criacao = models.DateTimeField(auto_now_add=True)
    data_processamento = models.DateTimeField(null=True, blank=True)
    erro_mensagem = models.TextField(blank=True)
    user = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
        blank=True
    )
    modelo = models.ForeignKey(
        'models.Modelo',
        on_delete=models.SET_NULL,
        null=True,
        blank=True
    )
    
    class Meta:
        verbose_name = "Item da Fila de Email"
        verbose_name_plural = "Fila de Emails"
        ordering = ['prioridade', 'data_agendamento']
        indexes = [
            models.Index(fields=['status', 'data_agendamento']),
            models.Index(fields=['prioridade', 'data_agendamento']),
        ]
    
    def __str__(self):
        return f"{self.destinatario_email} - {self.template.nome} ({self.get_status_display()})"