"""
Comando para testar o sistema de comunicação
"""
from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from communication.models import EmailTemplate, EmailQueue
from communication.services import EmailService, TriggerService, QueueProcessor
from models.models import Modelo
import logging

logger = logging.getLogger(__name__)


class Command(BaseCommand):
    help = 'Testa o sistema de comunicação enviando emails de exemplo'

    def add_arguments(self, parser):
        parser.add_argument(
            '--email',
            type=str,
            help='Email para enviar o teste',
            default='teste@allure.com.br'
        )
        parser.add_argument(
            '--template',
            type=str,
            help='Tipo do template para testar',
            default='cadastro_sucesso'
        )
        parser.add_argument(
            '--process-queue',
            action='store_true',
            help='Processar fila após enviar email'
        )

    def handle(self, *args, **options):
        email_teste = options['email']
        template_tipo = options['template']
        processar_fila = options['process_queue']
        
        self.stdout.write(f'Testando sistema de comunicação...')
        self.stdout.write(f'Email: {email_teste}')
        self.stdout.write(f'Template: {template_tipo}')
        
        try:
            # Buscar template
            template = EmailTemplate.objects.filter(
                tipo=template_tipo,
                ativo=True
            ).first()
            
            if not template:
                self.stdout.write(
                    self.style.ERROR(f'Template {template_tipo} não encontrado!')
                )
                return
            
            # Contexto de exemplo
            contexto = {
                'nome': 'Maria Silva',
                'email': email_teste,
                'data_cadastro': '15/01/2024',
                'modelo_id': 123,
                'site_url': 'https://allure.com.br',
                'site_nome': 'Allure',
                'data_atual': '15/01/2024',
                'hora_atual': '14:30',
            }
            
            # Adicionar variáveis específicas do template
            for variavel in template.get_variaveis():
                if variavel not in contexto:
                    contexto[variavel] = f'[Teste: {variavel}]'
            
            self.stdout.write('Enviando email de teste...')
            
            # Enviar email
            email_log = EmailService.send_email(
                template=template,
                destinatario_email=email_teste,
                contexto=contexto,
                destinatario_nome='Maria Silva'
            )
            
            if email_log.status == 'enviado':
                self.stdout.write(
                    self.style.SUCCESS(f'✅ Email enviado com sucesso!')
                )
                self.stdout.write(f'   Log ID: {email_log.id}')
                self.stdout.write(f'   Assunto: {email_log.assunto}')
            else:
                self.stdout.write(
                    self.style.ERROR(f'❌ Erro ao enviar email: {email_log.erro_mensagem}')
                )
            
            # Testar fila
            self.stdout.write('\nTestando sistema de fila...')
            
            # Adicionar email à fila
            queue_item = EmailService.queue_email(
                template=template,
                destinatario_email=email_teste,
                contexto=contexto,
                destinatario_nome='Maria Silva (Fila)',
                prioridade=1
            )
            
            self.stdout.write(f'✅ Email adicionado à fila (ID: {queue_item.id})')
            
            # Processar fila se solicitado
            if processar_fila:
                self.stdout.write('Processando fila...')
                stats = QueueProcessor.process_queue(batch_size=5)
                
                self.stdout.write(f'   Processados: {stats["processados"]}')
                self.stdout.write(f'   Enviados: {stats["enviados"]}')
                self.stdout.write(f'   Falharam: {stats["falharam"]}')
                
                if stats['erros']:
                    self.stdout.write(f'   Erros: {stats["erros"]}')
            
            # Testar gatilho
            self.stdout.write('\nTestando gatilho...')
            
            # Simular evento
            contexto_gatilho = {
                'nome': 'João Silva',
                'email': email_teste,
                'evento_teste': 'valor_teste'
            }
            
            queue_items = TriggerService.process_trigger(
                'modelo_cadastrado',
                contexto_gatilho
            )
            
            self.stdout.write(f'✅ Gatilho processado: {len(queue_items)} emails na fila')
            
            # Estatísticas gerais
            self.stdout.write('\n📊 Estatísticas:')
            total_templates = EmailTemplate.objects.count()
            templates_ativos = EmailTemplate.objects.filter(ativo=True).count()
            fila_pendente = EmailQueue.objects.filter(status='pendente').count()
            
            self.stdout.write(f'   Templates: {total_templates} total, {templates_ativos} ativos')
            self.stdout.write(f'   Fila pendente: {fila_pendente} emails')
            
            self.stdout.write(
                self.style.SUCCESS('\n🎉 Teste concluído com sucesso!')
            )
            
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro durante o teste: {str(e)}')
            )
            logger.error(f"Erro no teste de comunicação: {str(e)}")
