"""
Interface administrativa para o sistema de comunicação
"""
from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils import timezone
from django.db.models import Count, Q
from django.contrib.admin import SimpleListFilter

from .models import EmailTemplate, EmailLog, EmailTrigger, EmailQueue


class TemplateAtivoFilter(SimpleListFilter):
    title = 'Status do Template'
    parameter_name = 'ativo'
    
    def lookups(self, request, model_admin):
        return (
            ('ativo', 'Ativo'),
            ('inativo', 'Inativo'),
        )
    
    def queryset(self, request, queryset):
        if self.value() == 'ativo':
            return queryset.filter(ativo=True)
        elif self.value() == 'inativo':
            return queryset.filter(ativo=False)


@admin.register(EmailTemplate)
class EmailTemplateAdmin(admin.ModelAdmin):
    list_display = [
        'nome', 'tipo', 'assunto', 'ativo', 'emails_enviados', 
        'data_criacao', 'criado_por'
    ]
    list_filter = [TemplateAtivoFilter, 'tipo', 'data_criacao']
    search_fields = ['nome', 'assunto', 'tipo']
    readonly_fields = ['data_criacao', 'data_atualizacao']
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('nome', 'tipo', 'ativo')
        }),
        ('Conteúdo', {
            'fields': ('assunto', 'conteudo_html', 'conteudo_texto')
        }),
        ('Configurações', {
            'fields': ('variaveis_disponiveis',),
            'classes': ('collapse',)
        }),
        ('Metadados', {
            'fields': ('criado_por', 'data_criacao', 'data_atualizacao'),
            'classes': ('collapse',)
        }),
    )
    
    def emails_enviados(self, obj):
        count = obj.logs.filter(status='enviado').count()
        if count > 0:
            url = reverse('admin:communication_emaillog_changelist')
            return format_html(
                '<a href="{}?template__id__exact={}">{} emails</a>',
                url, obj.id, count
            )
        return '0 emails'
    emails_enviados.short_description = 'Emails Enviados'
    
    def save_model(self, request, obj, form, change):
        if not change:  # Novo template
            obj.criado_por = request.user
        super().save_model(request, obj, form, change)


class StatusFilter(SimpleListFilter):
    title = 'Status'
    parameter_name = 'status'
    
    def lookups(self, request, model_admin):
        return (
            ('enviado', 'Enviado'),
            ('falhou', 'Falhou'),
            ('pendente', 'Pendente'),
            ('entregue', 'Entregue'),
        )
    
    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(status=self.value())


@admin.register(EmailLog)
class EmailLogAdmin(admin.ModelAdmin):
    list_display = [
        'destinatario_email', 'assunto', 'template', 'status', 
        'data_envio', 'data_entrega', 'tempo_entrega'
    ]
    list_filter = [StatusFilter, 'template__tipo', 'data_envio']
    search_fields = ['destinatario_email', 'assunto', 'template__nome']
    readonly_fields = [
        'template', 'destinatario_email', 'destinatario_nome', 
        'assunto', 'status', 'data_envio', 'data_entrega', 
        'erro_mensagem', 'contexto', 'user', 'modelo'
    ]
    date_hierarchy = 'data_envio'
    
    def tempo_entrega(self, obj):
        if obj.data_entrega and obj.data_envio:
            delta = obj.data_entrega - obj.data_envio
            return f"{delta.total_seconds():.1f}s"
        return '-'
    tempo_entrega.short_description = 'Tempo de Entrega'
    
    def has_add_permission(self, request):
        return False  # Logs são criados automaticamente
    
    def has_change_permission(self, request, obj=None):
        return False  # Logs não devem ser editados


@admin.register(EmailTrigger)
class EmailTriggerAdmin(admin.ModelAdmin):
    list_display = [
        'nome', 'evento', 'template', 'ativo', 'delay_minutos', 
        'emails_disparados', 'data_criacao'
    ]
    list_filter = ['ativo', 'evento', 'data_criacao']
    search_fields = ['nome', 'evento', 'template__nome']
    readonly_fields = ['data_criacao', 'data_atualizacao']
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('nome', 'evento', 'template', 'ativo')
        }),
        ('Configurações', {
            'fields': ('delay_minutos', 'condicoes')
        }),
        ('Metadados', {
            'fields': ('data_criacao', 'data_atualizacao'),
            'classes': ('collapse',)
        }),
    )
    
    def emails_disparados(self, obj):
        count = EmailQueue.objects.filter(
            template=obj.template,
            status='enviado'
        ).count()
        if count > 0:
            url = reverse('admin:communication_emailqueue_changelist')
            return format_html(
                '<a href="{}?template__id__exact={}">{} emails</a>',
                url, obj.template.id, count
            )
        return '0 emails'
    emails_disparados.short_description = 'Emails Disparados'


class QueueStatusFilter(SimpleListFilter):
    title = 'Status da Fila'
    parameter_name = 'queue_status'
    
    def lookups(self, request, model_admin):
        return (
            ('pendente', 'Pendente'),
            ('processando', 'Processando'),
            ('enviado', 'Enviado'),
            ('falhou', 'Falhou'),
        )
    
    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(status=self.value())


@admin.register(EmailQueue)
class EmailQueueAdmin(admin.ModelAdmin):
    list_display = [
        'destinatario_email', 'template', 'status', 'prioridade', 
        'tentativas', 'data_agendamento', 'data_criacao'
    ]
    list_filter = [QueueStatusFilter, 'prioridade', 'template__tipo', 'data_criacao']
    search_fields = ['destinatario_email', 'template__nome']
    readonly_fields = [
        'template', 'destinatario_email', 'destinatario_nome', 
        'contexto', 'status', 'tentativas', 'data_criacao', 
        'data_processamento', 'erro_mensagem', 'user', 'modelo'
    ]
    date_hierarchy = 'data_criacao'
    actions = ['processar_fila', 'reprocessar_falhas']
    
    def processar_fila(self, request, queryset):
        from .services import QueueProcessor
        
        stats = QueueProcessor.process_queue(batch_size=queryset.count())
        
        self.message_user(
            request,
            f"Fila processada: {stats['processados']} processados, "
            f"{stats['enviados']} enviados, {stats['falharam']} falharam"
        )
    processar_fila.short_description = "Processar itens selecionados"
    
    def reprocessar_falhas(self, request, queryset):
        from .services import QueueProcessor
        
        reprocessados = QueueProcessor.retry_failed_emails()
        
        self.message_user(
            request,
            f"{reprocessados} emails reprocessados"
        )
    reprocessar_falhas.short_description = "Reprocessar falhas"
    
    def has_add_permission(self, request):
        return False  # Itens da fila são criados automaticamente


# Personalização do admin
admin.site.site_header = "Allure - Sistema de Comunicação"
admin.site.site_title = "Allure Admin"
admin.site.index_title = "Painel de Controle"

# Os modelos são registrados no admin_registry.py