from django.test import TestCase, Client
from django.urls import reverse
from django.contrib.auth.models import User
from django.core.files.uploadedfile import SimpleUploadedFile
from django.db import IntegrityError
from models.models import Modelo
from core.forms import ModeloRegistrationForm


class AccountsViewsTestCase(TestCase):
    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_login_view_get(self):
        """Testa a página de login (GET)"""
        response = self.client.get(reverse('accounts:login'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'accounts/login.html')

    def test_login_view_post_valid(self):
        """Testa login com credenciais válidas"""
        data = {
            'username': 'testuser',
            'password': 'testpass123',
        }
        response = self.client.post(reverse('accounts:login'), data)
        self.assertEqual(response.status_code, 302)  # Redirecionamento após login

    def test_login_view_post_invalid(self):
        """Testa login com credenciais inválidas"""
        data = {
            'username': 'testuser',
            'password': 'wrongpassword',
        }
        response = self.client.post(reverse('accounts:login'), data)
        self.assertEqual(response.status_code, 200)  # Volta para o formulário
        self.assertContains(response, 'Usuário ou senha inválidos')

    def test_logout_view(self):
        """Testa logout"""
        self.client.login(username='testuser', password='testpass123')
        response = self.client.get(reverse('accounts:logout'))
        self.assertEqual(response.status_code, 302)  # Redirecionamento após logout

    def test_cadastro_modelo_view_get(self):
        """Testa a página de cadastro de modelo (GET)"""
        response = self.client.get(reverse('accounts:cadastro_modelo'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'accounts/cadastro_modelo.html')

    def test_cadastro_modelo_view_post(self):
        """Testa cadastro de modelo com dados válidos"""
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        data = {
            'username': 'novomodelo',
            'email': 'novo@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Novo Modelo',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'sobre_mim': 'Nova descrição',
            'aceito_termos': True,
        }
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        response = self.client.post(reverse('accounts:cadastro_modelo'), data, files=files)
        self.assertEqual(response.status_code, 302)  # Redirecionamento após sucesso
        
        # Verificar se o modelo foi criado
        novo_modelo = Modelo.objects.filter(nome_exibicao='Novo Modelo').first()
        self.assertIsNotNone(novo_modelo)
        self.assertEqual(novo_modelo.nome_exibicao, 'Novo Modelo')

    def test_cadastro_modelo_view_post_invalid(self):
        """Testa cadastro com dados inválidos"""
        data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'differentpassword',  # Senhas diferentes
            'nome_exibicao': 'Modelo Teste',
            'idade': 16,  # Idade inválida
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': False,  # Não aceitou termos
        }
        
        response = self.client.post(reverse('accounts:cadastro_modelo'), data)
        self.assertEqual(response.status_code, 200)  # Volta para o formulário

    def test_editar_perfil_view_get(self):
        """Testa a página de edição de perfil (GET)"""
        self.client.login(username='testuser', password='testpass123')
        response = self.client.get(reverse('accounts:editar_perfil'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'accounts/editar_perfil.html')

    def test_editar_perfil_view_post(self):
        """Testa edição de perfil com dados válidos"""
        self.client.login(username='testuser', password='testpass123')
        
        # Criar modelo para o usuário
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Original',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            sobre_mim='Descrição original',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        data = {
            'nome_exibicao': 'Modelo Atualizado',
            'idade': 26,
            'estado': 'RJ',
            'cidade': 'Rio de Janeiro',
            'sobre_mim': 'Descrição atualizada',
            'categoria': 'FAN',
            'etnia': 'parda',
            'tipo_ensaio': 'dupla',
            'local_atendimento': 'motel',
        }
        
        response = self.client.post(reverse('accounts:editar_perfil'), data)
        self.assertEqual(response.status_code, 302)  # Redirecionamento após sucesso
        
        # Verificar se os dados foram atualizados
        modelo.refresh_from_db()
        self.assertEqual(modelo.nome_exibicao, 'Modelo Atualizado')
        self.assertEqual(modelo.idade, 26)
        self.assertEqual(modelo.estado, 'RJ')

    def test_esqueci_senha_view_get(self):
        """Testa a página de recuperação de senha (GET)"""
        response = self.client.get(reverse('accounts:esqueci_senha'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'accounts/esqueci_senha.html')

    def test_esqueci_senha_view_post(self):
        """Testa o envio de email de recuperação de senha"""
        data = {
            'email': 'test@example.com',
        }
        response = self.client.post(reverse('accounts:esqueci_senha'), data)
        self.assertEqual(response.status_code, 200)  # Página de confirmação


class AccountsFormsTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_cadastro_modelo_form_valid(self):
        """Testa formulário de cadastro válido"""
        from core.forms import ModeloRegistrationForm
        
        form_data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Novo Modelo',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'sobre_mim': 'Nova descrição',
            'aceito_termos': True,
        }
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        form = ModeloRegistrationForm(data=form_data, files=files)
        self.assertTrue(form.is_valid())

    def test_cadastro_modelo_form_invalid_email(self):
        """Testa formulário com email já existente"""
        from core.forms import ModeloRegistrationForm
        
        form_data = {
            'username': 'testuser2',
            'email': 'test@example.com',  # Email já existe
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Novo Modelo',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': True,
        }
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        form = ModeloRegistrationForm(data=form_data, files=files)
        self.assertFalse(form.is_valid())
        self.assertIn('email', form.errors)


class AccountsModelsTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_modelo_user_relationship(self):
        """Testa relacionamento entre usuário e modelo"""
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        # Verificar relacionamento
        self.assertEqual(modelo.user, self.user)
        self.assertEqual(self.user.modelo_profile, modelo)

    def test_modelo_creation(self):
        """Testa criação de modelo"""
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        self.assertEqual(modelo.nome_exibicao, 'Modelo Teste')
        self.assertEqual(modelo.status, 'pendente')  # Status padrão
        self.assertFalse(modelo.is_boost_ativo)  # Sem plano ativo por padrão

    def test_modelo_str_representation(self):
        """Testa representação string do modelo"""
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        self.assertEqual(str(modelo), 'Modelo Teste')


class AccountsIntegrationTestCase(TestCase):
    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_complete_user_workflow(self):
        """Testa o fluxo completo de cadastro, login e uso do sistema"""
        # 1. Cadastrar novo modelo
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        data = {
            'username': 'novomodelo',
            'email': 'novo@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Modelo Completo',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'sobre_mim': 'Descrição completa',
            'aceito_termos': True,
        }
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        response = self.client.post(reverse('accounts:cadastro_modelo'), data, files=files)
        self.assertEqual(response.status_code, 302)
        
        # 2. Fazer login
        login_data = {
            'username': 'novomodelo',
            'password': 'testpass123',
        }
        response = self.client.post(reverse('accounts:login'), login_data)
        self.assertEqual(response.status_code, 302)
        
        # 3. Acessar perfil
        response = self.client.get(reverse('accounts:editar_perfil'))
        self.assertEqual(response.status_code, 200)

    def test_authentication_required_views(self):
        """Testa que views protegidas requerem autenticação"""
        # Tentar acessar editar perfil sem login
        response = self.client.get(reverse('accounts:editar_perfil'))
        self.assertEqual(response.status_code, 302)  # Redirecionamento para login


class AccountsSecurityTestCase(TestCase):
    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_csrf_protection_login(self):
        """Testa proteção CSRF no login"""
        data = {
            'username': 'testuser',
            'password': 'testpass123',
        }
        
        # Usar client sem CSRF para testar
        client = Client(enforce_csrf_checks=True)
        response = client.post(reverse('accounts:login'), data)
        self.assertEqual(response.status_code, 403)  # CSRF error

    def test_password_validation(self):
        """Testa validação de senha"""
        from core.forms import ModeloRegistrationForm
        
        form_data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': '123',  # Senha muito curta
            'password2': '123',
            'nome_exibicao': 'Modelo Teste',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': True,
        }
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        form = ModeloRegistrationForm(data=form_data, files=files)
        self.assertFalse(form.is_valid())
        self.assertIn('password1', form.errors)

    def test_session_security(self):
        """Testa segurança da sessão"""
        # Login
        self.client.login(username='testuser', password='testpass123')
        
        # Verificar se está logado
        response = self.client.get(reverse('accounts:editar_perfil'))
        self.assertEqual(response.status_code, 200)
        
        # Logout
        self.client.logout()
        
        # Verificar se não está mais logado
        response = self.client.get(reverse('accounts:editar_perfil'))
        self.assertEqual(response.status_code, 302)  # Redirecionamento para login
